/* SCE CONFIDENTIAL
 * PlayStation(R)3 Programmer Tool Runtime Library 475.001
 * Copyright (C) 2011 Sony Computer Entertainment Inc.
 * All Rights Reserved.
 */


#include <cell/spurs.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

#include "sample_vision_util_spurs.h"
#include "sample_vision_util_task.h"

#include "cellImage.h"
#include "vision_param.h"
#include "sample_vision_elf_binary_ext.h"
#include "memory.h"
#include "TemplateMatchingInfo.h"
#include "TemplateMatching.h"

#define VERVOSE_DEBUG 0

#define max(a, b) (((a) > (b)) ? (a) : (b))


/******************************************************************************
 * TemplateMatching (constructor)
 *****************************************************************************/
TemplateMatching::
TemplateMatching()
{
	mInit = false;
}

/******************************************************************************
 * ~TemplateMatching (destructor)
 *****************************************************************************/
TemplateMatching::
~TemplateMatching()
{
	
}

/******************************************************************************
 * GetAlign()
 *****************************************************************************/
int TemplateMatching::
GetAlign(void) {
	int align = TEMPLATE_MATCHING_ALIGN;

	align = max(align, sampleVisionUtilTask2GetAlign());
	align = max(align, CELL_SPURS_TASKSET_ALIGN);
	align = max(align, TemplateMatchingInfo::GetAlign());
	align = max(align, TEMPLATE_MATCHING_RESULT_ALIGN);

	return align;
}

/******************************************************************************
 * GetWorkingMemorySize()
 *****************************************************************************/
int TemplateMatching::
GetWorkingMemorySize(int size) {
	int memsize = size;
	
	return (memsize - size);
}


/******************************************************************************
 * Initialize()
 *****************************************************************************/
int TemplateMatching::
Initialize(const void *work, const CellSpurs2 *spurs)
{
	if (mInit) return TEMPLATE_MATCHING_ERROR;
	
	unsigned char *p = (unsigned char*)work;
	
	////////////////////////////////////////
	// spurs taskset initialize
	////////////////////////////////////////
	const unsigned char priorities[8] = {1, 1, 1, 1, 1, 1, 1, 1};
	int ret = sampleVisionUtilInitializeEx((CellSpurs2*)spurs, &mSpursTaskset, priorities, 2);
	if (ret != CELL_OK) return ret;
	
	////////////////////////////////////////
	//memory assignment
	////////////////////////////////////////
	mResult.mPosX = 0.0f;
	mResult.mPosY = 0.0f;
	mResult.mScore = 0.0f;
	for(int i=0; i<TM_TEMPLATE_NUM_MAX; i++) {
		mTemplateResult[i].mPosX = 0.0f;
		mTemplateResult[i].mPosY = 0.0f;
		mTemplateResult[i].mScore = 0.0f;
	}
	
	//set initialize flag
	mInit = true;

	return (int)(p - (unsigned char*)work);
}

/******************************************************************************
 * Finalize()
 *****************************************************************************/
int TemplateMatching::
Finalize(void)
{
	if (!mInit) return TEMPLATE_MATCHING_ERROR;

	int ret = sampleVisionUtilFinalizeEx(&mSpursTaskset);
	if (ret != CELL_OK) return ret;
	
	mInit = false;
	
	return TEMPLATE_MATCHING_OK;
}


/******************************************************************************
 * TemplateMatching()
 * tplate : template [num * tplate_size^2]
 * tplate_num : 0-(tplate_num-1):base+motion template, tplate_num-1:prev template
 * xpos, ypos : searching point center (center of template)
 *****************************************************************************/
int TemplateMatching::
ExecTemplateMatching(const CellImage *image, const TemplateMatchingInfo *tmInfo)
{
	if (!mInit) return TEMPLATE_MATCHING_ERROR;
	
	///////////////////////////////////////////
	//calculate matching score
	// base , update , prev
	///////////////////////////////////////////
	tmInfo->GetPosition(&mBMInfo.ix, &mBMInfo.iy);
	//change from center to upper-left position
	mBMInfo.ix -= (float)TM_TEMPLATE_SIZE / 2.0f;
	mBMInfo.iy -= (float)TM_TEMPLATE_SIZE / 2.0f;

	for(int num=0; num<TM_TEMPLATE_NUM_MAX; num++) {

		mBMInfo.ox = 0.0f;
		mBMInfo.oy = 0.0f;
		mBMInfo.score = 0.0f;
		mBMInfo.roll = 0.0f;

		int ret;
		ret = sampleVisionUtilBlockMatching16x16(
			&mSpursTaskset,
			&mBlockMatchingTask,
			image,
			tmInfo->GetTemplate((TM_TEMPLATE_NUM)num),
			&mBMInfo);

		if( ret != CELL_OK ) { //error
			mTemplateResult[num].mPosX = mBMInfo.ox + (float)TM_TEMPLATE_SIZE / 2.0f;
			mTemplateResult[num].mPosY = mBMInfo.oy + (float)TM_TEMPLATE_SIZE / 2.0f;;
			mTemplateResult[num].mScore = -1.0f; //error
		}
		else {
			//change from upper-left to center position
			mTemplateResult[num].mPosX = mBMInfo.ox + (float)TM_TEMPLATE_SIZE / 2.0f;
			mTemplateResult[num].mPosY = mBMInfo.oy + (float)TM_TEMPLATE_SIZE / 2.0f;;
			mTemplateResult[num].mScore = mBMInfo.score;
		}
			
		

#if VERVOSE_DEBUG
	fprintf(stderr, "#TemplateMatching::ExecTemplateMatching() [%d] (%f, %f) -> (%f, %f) %f\n",
			num, mBMInfo.ix, mBMInfo.iy,
			mTemplateResult[num].mPosX, mTemplateResult[num].mPosY,
			mTemplateResult[num].mScore);
#endif //VERVOSE_DEBUG
		
	}

	///////////////////////////////////////////
	//sort base template score
	///////////////////////////////////////////
	float maxScore1 = 0.0f; //max score
	float maxScore2 = 0.0f; //secondary max score
	int maxNum1 = 0; //template number of max score
	int maxNum2 = 0; //template number of secondary max score
	for(int num=0; num<=TM_TEMPLATE_RIGHT_ROLL; num++) { //only base tp loop
		float score = mTemplateResult[num].mScore;
		if(maxScore2 < score) { //
			if(maxScore1 < score) { //max score
				maxScore2 = maxScore1;
				maxNum2 = maxNum1;
				maxScore1 = score;
				maxNum1 = num;
			}
			else { //secondary max score
				maxScore2 = score;
				maxNum2 = num;
			}
		}
	}

#if VERVOSE_DEBUG
	fprintf(stderr, "#TemplateMatching::ExecTemplateMatching() maxscore=%f, %f, num=%d, %d (thre=%f, %f)\n",
			maxScore1, maxScore2, maxNum1, maxNum2,
			TM_SCORE_THRE, TM_SCORE_THRE_BASE);
#endif //VERVOSE_DEBUG

	/////////////////////////////////////////////
	// base template result
	/////////////////////////////////////////////
	float weight1;
	float weight2;
	if(maxScore1 >= TM_SCORE_THRE) { //match base tp
		float scoreDif1 = maxScore1 - TM_SCORE_THRE;
		float scoreDif2;
		
		if(maxScore2 >= TM_SCORE_THRE) { //weighted result
			scoreDif2 = maxScore2 - TM_SCORE_THRE;
			weight1 = scoreDif1 / (scoreDif1 + scoreDif2);
			weight2 = scoreDif2 / (scoreDif1 + scoreDif2);
			mResult.mScore = mTemplateResult[maxNum1].mScore * weight1 +
				mTemplateResult[maxNum2].mScore * weight2;
			mResult.mPosX = mTemplateResult[maxNum1].mPosX * weight1 +
				mTemplateResult[maxNum2].mPosX * weight2;
			mResult.mPosY = mTemplateResult[maxNum1].mPosY * weight1 +
				mTemplateResult[maxNum2].mPosY * weight2;
		}
		else {
			mResult.mScore = mTemplateResult[maxNum1].mScore;
			mResult.mPosX = mTemplateResult[maxNum1].mPosX;
			mResult.mPosY = mTemplateResult[maxNum1].mPosY;
		}
		mUsedTemplate = maxNum1;
	}
	/////////////////////////////////////////////
	// additional template result
	/////////////////////////////////////////////
	else if(mTemplateResult[TM_TEMPLATE_ADDITIONAL].mScore >= TM_SCORE_THRE) { //match additional tp
		if(maxScore1 >= TM_SCORE_THRE_BASE) { //base >= thre_base
			weight1 = (maxScore1 - TM_SCORE_THRE_BASE) /
				(TM_SCORE_THRE - TM_SCORE_THRE_BASE);
			weight2 = 1.0 - weight1;

			mResult.mScore = mTemplateResult[TM_TEMPLATE_ADDITIONAL].mScore;
			mResult.mPosX = mTemplateResult[maxNum1].mPosX * weight1 +
				mTemplateResult[TM_TEMPLATE_ADDITIONAL].mPosX * weight2;
			mResult.mPosY = mTemplateResult[maxNum1].mPosY * weight1 +
				mTemplateResult[TM_TEMPLATE_ADDITIONAL].mPosY * weight2;
			mUsedTemplate = TM_TEMPLATE_ADDITIONAL;
		}
		else {
			mResult.mScore = mTemplateResult[TM_TEMPLATE_ADDITIONAL].mScore;
			mResult.mPosX = mTemplateResult[TM_TEMPLATE_ADDITIONAL].mPosX;
			mResult.mPosY = mTemplateResult[TM_TEMPLATE_ADDITIONAL].mPosY;
			mUsedTemplate = TM_TEMPLATE_ADDITIONAL;
		}
	}
	/////////////////////////////////////////////
	// prev template result
	/////////////////////////////////////////////
	else if(mTemplateResult[TM_TEMPLATE_PREVIOUS].mScore >= TM_SCORE_THRE) { //match prev tp
		if(maxScore1 >= TM_SCORE_THRE_BASE) { //base >= thre_base
			weight1 = (maxScore1 - TM_SCORE_THRE_BASE) /
				(TM_SCORE_THRE - TM_SCORE_THRE_BASE);
			weight2 = 1.0 - weight1;

			mResult.mScore = mTemplateResult[TM_TEMPLATE_PREVIOUS].mScore;
			mResult.mPosX = mTemplateResult[maxNum1].mPosX * weight1 +
				mTemplateResult[TM_TEMPLATE_PREVIOUS].mPosX * weight2;
			mResult.mPosY = mTemplateResult[maxNum1].mPosY * weight1 +
				mTemplateResult[TM_TEMPLATE_PREVIOUS].mPosY * weight2;
			mUsedTemplate = TM_TEMPLATE_PREVIOUS;
		}
		else {
			mResult.mScore = mTemplateResult[TM_TEMPLATE_PREVIOUS].mScore;
			mResult.mPosX = mTemplateResult[TM_TEMPLATE_PREVIOUS].mPosX;
			mResult.mPosY = mTemplateResult[TM_TEMPLATE_PREVIOUS].mPosY;
			mUsedTemplate = TM_TEMPLATE_PREVIOUS;
		}
	}
	else { //tentative result (not match)
		mResult.mScore = mTemplateResult[maxNum1].mScore; //smaller than threshold
		mResult.mPosX = mTemplateResult[maxNum1].mPosX;
		mResult.mPosY = mTemplateResult[maxNum1].mPosY;
		mUsedTemplate = TM_TEMPLATE_NUM_MAX;

		return TEMPLATE_MATCHING_ERROR;
	}
	
	return TEMPLATE_MATCHING_OK;
}


