/* SCE CONFIDENTIAL
 * PlayStation(R)3 Programmer Tool Runtime Library 475.001
 * Copyright (C) 2011 Sony Computer Entertainment Inc.
 * All Rights Reserved.
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <cell/face.h>
#include <cell/spurs.h>

#include "memory.h"

#include <new>

#include "sample_face_util.h"
#include "TargetInfo.h"
#include "LocalSearchInfo.h"
#include "LocalFaceSearch.h"


#define VERVOSE_DEBUG 0


#define max(a, b) (((a) > (b)) ? (a) : (b))
#define min(a, b) (((a) < (b)) ? (a) : (b))

////////////////////////////////////////////////
// Constructor
////////////////////////////////////////////////
LocalFaceSearch::
LocalFaceSearch()
{
	mInit = false;
	mWorkingArea = NULL;
}

////////////////////////////////////////////////
// Destructor
////////////////////////////////////////////////
LocalFaceSearch::
~LocalFaceSearch()
{

}

////////////////////////////////////////////////
// GetAlign
////////////////////////////////////////////////
int LocalFaceSearch::
GetAlign(void)
{
	int align = LOCAL_FACE_SEARCH_ALIGN;

	align = max(align, CELL_SPURS_TASKSET_ALIGN);
	align = max(align, sampleFaceUtilTask2GetAlign());
	align = max(align, 16); //compare CellFace...
	
	return align;
}

////////////////////////////////////////////////
// GetWorkingMemorySize
// static member function
// calc additional memory size
////////////////////////////////////////////////
int LocalFaceSearch::
GetWorkingMemorySize(int size, int width, int height, int rowstride, int maxTarget)
{
	(void)width;
	(void)maxTarget;
	int memsize = size;

	memsize += sizeof_byte_align(128, memsize, CELL_FACE_UTIL_WORK_SIZE(width, height, rowstride));

	return (memsize - size);
}

////////////////////////////////////////////////
// Initialize
// instantiation is necessery before call this function
////////////////////////////////////////////////
int LocalFaceSearch::
Initialize(const void *work, const CellSpurs2 *spurs, int width, int height, int rowstride, int maxTarget)
{
	if (mInit) return LOCAL_FACE_SEARCH_ERROR;
	
	unsigned char *p = (unsigned char*)work;

	////////////////////////////////////////
	// spurs taskset initialize
	////////////////////////////////////////
	const unsigned char priorities[8] = {1, 1, 1, 1, 1, 1, 1, 1};
	int ret = sampleFaceUtilInitializeEx((CellSpurs2*)spurs, &mSpursTaskset, priorities, 2);
	if (ret != CELL_OK) return ret;
	
	////////////////////////////////////////
	//memory assignment
	////////////////////////////////////////
	p += assign_memory_byte_align(128, (void *)p, (void *)&mWorkingArea, CELL_FACE_UTIL_WORK_SIZE(width, height, rowstride));

	/////////////////////////////////////
	//set parameter
	/////////////////////////////////////
	mWidth = width;
	mHeight = height;
	mRowstride = rowstride;
	mMaxTarget = maxTarget;
	mDetectPartsFlag = false;
	mOverlapFlag = false;
	
	//set initialize flag
	mInit = true;
	
	/////////////////////////////////////
	// set libface parameter
	/////////////////////////////////////
	//local search parameter set
	cellFaceUtilDetection3DParamInitialize(
		&mDetectParam,
		NULL, mWidth, mHeight, mRowstride,
		mWorkingArea,
		mDetectResult, MAX_LOCAL_DETECT_NUM
	);

	mDetectData.eaFaceResult = NULL;
	mDetectData.numFace      = 0;
	cellFaceUtilOptParamLocalSearchInitialize(
		&mDetectOpt,
		&mDetectParam,
		&mDetectData, 1
	);
	
	mDetectOpt.scanFaceMarginX = LOCAL_FACE_SEARCH_SCAN_FACE_MARGIN_X;
	mDetectOpt.scanFaceMarginY = LOCAL_FACE_SEARCH_SCAN_FACE_MARGIN_Y;

	// initialize parameter for parts
	cellFaceUtilPartsParamInitialize(
		&mPartsParam,
		NULL, mWidth, mHeight, mRowstride,
		mWorkingArea,
		mDetectResult,
		mPartsResult, &mPosition );

	return (int)(p - (unsigned char*)work);
}

////////////////////////////////////////////////
// Finalize
////////////////////////////////////////////////
int LocalFaceSearch::
Finalize(void)
{
	if (!mInit) return LOCAL_FACE_SEARCH_ERROR;
	
	int ret = sampleFaceUtilFinalizeEx(&mSpursTaskset);
	if (ret != CELL_OK) return ret;

	mWorkingArea = NULL;
	
	mInit = false;
	
	return LOCAL_FACE_SEARCH_OK;
}


////////////////////////////////////////////////
// ExecFaceSearch
// not check target status
////////////////////////////////////////////////
int LocalFaceSearch::
ExecFaceSearch(const unsigned char *image, //input Y image
			   const int *targetStatus, //input [mMaxTarget]
			   const LocalSearchInfo *lsInfo, //input [mMaxTarget]
			   int target //input target
			   )
{
	if (!mInit) return LOCAL_FACE_SEARCH_ERROR;
	
	mDetectPartsFlag = false;
	mOverlapFlag = false;

	int numFace = 0;
	int ret = 0;

	mDetectParam.eaImage = (uintptr_t)image; //image
	//set parameter (base position)
	mDetectData.eaFaceResult = (uintptr_t)lsInfo[target].GetDetectResult();
	mDetectData.numFace = 1;

	//execute local search
	ret = sampleFaceUtilDetection3DTaskBegin(&mSpursTaskset, &mDetectTask, &mDetectParam);
	if (ret != CELL_FACE_OK) { //local search error
		return ret;
	}
	ret = sampleFaceUtilDetection3DTaskEnd(&mSpursTaskset, &mDetectTask, &numFace);
	if (ret != CELL_FACE_OK) { //local search error
		return ret;
	}

	int num; 
	for(num = 0; num < numFace; num++) { //detection face loop
		//////////////////////////////
		//self overlap check
		//////////////////////////////
		if (CheckOverlap(mDetectResult[num], *lsInfo[target].GetDetectResult(),
						 LOCAL_FACE_SEARCH_SELF_OVERLAP_THRE)) { //overlap
			//update current result
			memcpy(&mCurrResult, &mDetectResult[num], sizeof(CellFaceDetectionResult));
			break;
		} //overlap
	} //numFace loop

	if(num >= numFace) { //lost
		return LOCAL_FACE_SEARCH_ERROR;
	}

	//////////////////////////////
	//other target overlap check
	//ignore lost target
	//////////////////////////////
	for(int tgt = 0; tgt < mMaxTarget; tgt++) {
		if(tgt != target) {
			if(targetStatus[tgt] == TARGET_STATUS_TRACKING) {
				if (CheckOverlap(mDetectResult[num], *lsInfo[tgt].GetDetectResult(),
								 LOCAL_FACE_SEARCH_OTHER_OVERLAP_THRE)) { //overlap
					mOverlapFlag = true;
					return LOCAL_FACE_SEARCH_OK;
				}
			}
		} //tgt != target
	} //tgt loop
	
	//////////////////////////////
	//parts detection
	//////////////////////////////
	mPartsParam.eaImage = (uintptr_t)image;
	mPartsParam.faceX = mDetectResult[num].faceX;
	mPartsParam.faceY = mDetectResult[num].faceY;
	mPartsParam.faceW = mDetectResult[num].faceW;
	mPartsParam.faceH = mDetectResult[num].faceH;
	mPartsParam.faceRoll  = mDetectResult[num].faceRoll;
	mPartsParam.facePitch = mDetectResult[num].facePitch;
	mPartsParam.faceYaw   = mDetectResult[num].faceYaw;
	ret = sampleFaceUtilPartsTaskBegin(&mSpursTaskset, &mPartsTask, &mPartsParam);
	if(ret != CELL_FACE_OK) { //can't detect parts
		mDetectPartsFlag = false;
		return LOCAL_FACE_SEARCH_OK; //parts detection isn't ok, but local face search is ok
	}
	ret = sampleFaceUtilPartsTaskEnd(&mSpursTaskset, &mPartsTask);
	if (ret != CELL_FACE_OK) { //can't detect parts
		mDetectPartsFlag = false;
		return LOCAL_FACE_SEARCH_OK; //parts detection isn't ok, but local face search is ok
	}

#if VERVOSE_DEBUG
	for(int i=0; i<CELL_FACE_PARTS_NUM_MAX; i++) {
		fprintf(stderr, "#LocalFaceSearch::ExecFaceSearch() PartsResult[%d] : %d, %f, %f, %f\n",
				i, 
				mPartsResult[i].partsId,
				mPartsResult[i].partsX,
				mPartsResult[i].partsY,
				mPartsResult[i].score
				);
	}
#endif //VERVOSE_DEBUG
	
	mDetectPartsFlag = true;
	return LOCAL_FACE_SEARCH_OK;
}


/////////////////////////////////
// GetPositionResultXYCenter
/////////////////////////////////
void LocalFaceSearch::
GetPositionResultXYCenter(float *x, float *y) const
{
	*x = 0.0;
	*y = 0.0;
	for(int i=0; i<4; i++) {
		*x += mPosition.regionX[i];
		*y += mPosition.regionY[i];
	}
	
	*x *= ((float)mWidth / 4.0f);
	*y *= ((float)mHeight / 4.0f);
}

/////////////////////////////////
// GetPositionResultEyeCenter
/////////////////////////////////
void LocalFaceSearch::
GetPositionPartsEyeCenter(float *x, float *y) const
{
	float faceX = mCurrResult.faceX;
	float faceY = mCurrResult.faceY;
	float faceW = mCurrResult.faceW;
	float faceH = mCurrResult.faceH;
	
	float partsRX = mPartsResult[CELL_FACE_PARTS_ID_R_EYE_CENTER - 1].partsX;
	float partsRY = mPartsResult[CELL_FACE_PARTS_ID_R_EYE_CENTER - 1].partsY;
	float partsLX = mPartsResult[CELL_FACE_PARTS_ID_L_EYE_CENTER - 1].partsX;
	float partsLY = mPartsResult[CELL_FACE_PARTS_ID_L_EYE_CENTER - 1].partsY;
	float partsCX = partsRX + partsLX;
	float partsCY = partsRY + partsLY;
	
	*x = (faceX + partsCX * faceW / 2.0f) * mWidth;
	*y = (faceY + partsCY * faceH / 2.0f) * mHeight;
}

/////////////////////////////////
// GetPositionResultPose
/////////////////////////////////
void LocalFaceSearch::
GetPositionResultPose(float *yaw, float *pitch, float *roll) const
{
	*yaw = -mPosition.pose[2];
	*pitch = mPosition.pose[1];
	*roll = -mPosition.pose[0];
}



////////////////////////////////////////////////////////////////////////
//private
////////////////////////////////////////////////////////////////////////
/////////////////////////////////
// CheckOverlap
/////////////////////////////////
bool LocalFaceSearch::
CheckOverlap(
	const CellFaceDetectionResult& a,
	const CellFaceDetectionResult& b,
	const float thresh
)
{
	const float tmpx = b.faceX - a.faceX;
	const float tmpy = b.faceY - a.faceY;

	if (((a.faceW - tmpx)*(b.faceW + tmpx) > 0.0f) &&
	    ((a.faceH - tmpy)*(b.faceH + tmpy) > 0.0f)) {
		// Overlapped
		const float x1 = max(a.faceX, b.faceX);
		const float y1 = max(a.faceY, b.faceY);
		const float x2 = min(a.faceX + a.faceW, b.faceX + b.faceW);
		const float y2 = min(a.faceY + a.faceH, b.faceY + b.faceH);
		const float a_area   = a.faceW * a.faceH;
		const float b_area   = b.faceW * b.faceH;
		const float and_area = (x2 - x1) * (y2 - y1);

#if VERVOSE_DEBUG
		fprintf(stderr, "#ls overlapcheck a(%f,%f)-(%f,%f) b(%f,%f)-(%f,%f) score a(%f),b(%f)\n",
				a.faceX, a.faceY, a.faceX+a.faceW, a.faceY+a.faceH,
				b.faceX, b.faceY, b.faceX+b.faceW, b.faceY+b.faceH,
				and_area / a_area, and_area/ b_area);
#endif //VERVOSE_DEBUG
		
		return thresh < (and_area / a_area) || thresh < (and_area / b_area);
	}
	// Not overlapped
	return false;
}



