/* SCE CONFIDENTIAL
 * PlayStation(R)3 Programmer Tool Runtime Library 475.001
 * Copyright (C) 2011 Sony Computer Entertainment Inc.
 * All Rights Reserved.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

#include <cell/spurs.h>
#include <cell/face.h>
#include <cell/face_tracker.h>
#include "libface_tracker_internal.h"

#include <sys/sys_time.h>
#include <sys/time_util.h>
#include <sys/spu_initialize.h>

#include <new>

#include "FaceRegisterInfo.h"
#include "LocalSearchInfo.h"
#include "TemplateMatchingInfo.h"
#include "GlobalFaceSearch.h"
#include "LocalFaceSearch.h"
#include "TemplateMatching.h"
#include "cellImage.h"

#include "sample_vision_util_spurs.h"
#include "sample_vision_util_task.h"

#include "memory.h"


#define VERVOSE_DEBUG 0

#define max(a, b) (((a) > (b)) ? (a) : (b))


#define PARTS_LOG_LIKELIHOOD_THRESHOLD (-10.0f) //KOBAY

bool gCellFaceTrackerInitFlag = false; //initialized flag

CellFaceTrackerCallbackFunc gCellFaceTrackerCallbackFunc;

//CellFaceTracker
CellFaceTracker *gFT;
//result
CellFaceTrackerResult *gResult; //[mMaxTarget]

//time data
unsigned long long gRecogTimeTotal = 0ULL;
unsigned long long gTrackingTimeTotal = 0ULL;
unsigned long long gTimerStart = 0ULL;
unsigned long long gTimerEnd = 0ULL;


///////////////////////////////////////////
// cellFaceTrackerRegisterCallback
///////////////////////////////////////////
int cellFaceTrackerRegisterCallback(CellFaceTrackerCallbackFunc func)
{
	if(!gCellFaceTrackerInitFlag) {
		return CELL_FACE_TRACKER_ERROR_NOT_INITIALIZED;
	}

	gCellFaceTrackerCallbackFunc = func;
	//gFT->mGlobalSearch.RegisterCallback(func);
	
	return CELL_FACE_TRACKER_OK;
}


//////////////////////////////////////////
// cellFaceTrackerStartTracking
//////////////////////////////////////////
int cellFaceTrackerStartTracking(int id, const CellFaceFeature2 *feature)
{
	if(!gCellFaceTrackerInitFlag) {
		return CELL_FACE_TRACKER_ERROR_NOT_INITIALIZED;
	}
	if(id < 0) {
		return CELL_FACE_TRACKER_ERROR_INVALID_PARAM;
	}
	if(feature == NULL) {
		return CELL_FACE_TRACKER_ERROR_INVALID_PARAM;
	}

	for(int tgt=0; tgt<gFT->mMaxTarget; tgt++) {
		if(gFT->mTargetStatus[tgt] != TARGET_STATUS_NULL) { //occupied target
			continue;
		}
		else { //null target
			gFT->mTargetStatus[tgt] = TARGET_STATUS_EMPTY;
			gFT->mFaceRegisterInfo[tgt].RegisterFaceID(id, feature);
			
			// clear parameter
			gFT->mTrackCount[tgt] = 0;
			gFT->mLocalSearchInfo[tgt].ClearLostCount();
			gFT->mTemplateMatchingInfo[tgt].ClearLostCount();
			gFT->mLocalSearchTimingOffset[tgt] = 0;
			
			return CELL_FACE_TRACKER_OK;
		}
	}

	return CELL_FACE_TRACKER_ERROR_FACEID_FULL;
}


//////////////////////////////////////////
// cellFaceTrackerStopTracking
//////////////////////////////////////////
int cellFaceTrackerStopTracking(int id)
{
	if(!gCellFaceTrackerInitFlag) {
		return CELL_FACE_TRACKER_ERROR_NOT_INITIALIZED;
	}
	if(id < 0){
		return CELL_FACE_TRACKER_ERROR_INVALID_PARAM;
	}
	
	//select target id
	for(int tgt=0; tgt<gFT->mMaxTarget; tgt++) {
		if(gFT->mTargetStatus[tgt] == TARGET_STATUS_NULL) { //not registered target
			continue;
		}
		else {
			if(gFT->mFaceRegisterInfo[tgt].GetID() != id) { //different id
				continue;
			}
			else { //same id
				gFT->mTargetStatus[tgt] = TARGET_STATUS_NULL;
				gFT->mFaceRegisterInfo[tgt].UnregisterFaceID();
				
				// clear parameter
				gFT->mTrackCount[tgt] = 0;
				gFT->mLocalSearchInfo[tgt].ClearLostCount();
				gFT->mTemplateMatchingInfo[tgt].ClearLostCount();
				gFT->mLocalSearchTimingOffset[tgt] = 0;
				gFT->mTemplateMatchingInfo[tgt].ClearTemplateAll();
				
				return CELL_FACE_TRACKER_OK;
			}
		}
	}

	return CELL_FACE_TRACKER_ERROR_FACEID_NOT_EXIST;
}

///////////////////////////////////////////////
// cellFaceTrackerReset
///////////////////////////////////////////////
int cellFaceTrackerReset(void)
{
	if(!gCellFaceTrackerInitFlag) {
		return CELL_FACE_TRACKER_ERROR_NOT_INITIALIZED;
	}
	
	for(int tgt=0; tgt<gFT->mMaxTarget; tgt++) {
		if(gFT->mTargetStatus[tgt] == TARGET_STATUS_NULL) { //not registered target
			continue;
		}
		else {
			gFT->mTargetStatus[tgt] = TARGET_STATUS_NULL;
			gFT->mFaceRegisterInfo[tgt].UnregisterFaceID();
			
			// clear parameter
			gFT->mTrackCount[tgt] = 0;
			gFT->mLocalSearchInfo[tgt].ClearLostCount();
			gFT->mTemplateMatchingInfo[tgt].ClearLostCount();
			gFT->mLocalSearchTimingOffset[tgt] = 0;
			gFT->mTemplateMatchingInfo[tgt].ClearTemplateAll();
		}
	}
	gFT->mProcessCount = 0;
	
	return CELL_FACE_TRACKER_OK;
}

/////////////////////////////////////////////////////
// cellFaceTrackerGetWorkingMemorySize
// return values : error code
/////////////////////////////////////////////////////
int cellFaceTrackerGetWorkingMemorySize(
	int *worksize, //necessary memory size [Byte]
	int imageWidth, //image width
	int imageHeight, //image height
	int imageRowstride, //image rowstride
	int maxTrackingNum //max target number
	)
{
	
	if((imageRowstride < 0) ||
	   (imageRowstride > CELL_FACE_TRACKER_IMG_WIDTH_MAX)) {//imageRowstride check
		return CELL_FACE_TRACKER_ERROR_IMAGE_WIDTH_LIMIT;
	}
	if((imageRowstride&0x0000000f) != 0) { //imageRowstride 16Byte align check
		return CELL_FACE_TRACKER_ERROR_ROWSTRIDE_ALIGN;
	}
	if((imageWidth < 0) ||
	   (imageWidth > CELL_FACE_TRACKER_IMG_WIDTH_MAX)) { //imageWidth check
		return CELL_FACE_TRACKER_ERROR_IMAGE_WIDTH_LIMIT;
	}
	if(imageHeight < 0 ||
	   imageHeight > CELL_FACE_TRACKER_IMG_HEIGHT_MAX ||
	   maxTrackingNum < 0 ||
	   maxTrackingNum > CELL_FACE_DETECT_NUM_MAX) {
		return CELL_FACE_TRACKER_ERROR_INVALID_PARAM;
	}

	
	int memsize = 0;
	memsize += sizeof_byte_align(CELL_FACE_TRACKER_ALIGN, memsize,
								 sizeof(CellFaceTracker)); //for *gFT
	
	memsize += sizeof_byte_align(CELL_IMAGE_DATA_ALIGN, memsize,
								 sizeof(unsigned char) * (imageRowstride/2) * (imageHeight/2));
	memsize += sizeof_byte_align(CELL_IMAGE_DATA_ALIGN, memsize,
								 sizeof(unsigned char) * (imageRowstride/4) * (imageHeight/4));
	
	memsize += CellImage::GetWorkingMemorySize(memsize, imageRowstride, imageHeight,
										CELL_IMAGE_DATA_ALLOCATE_EXTERNAL);
	memsize += CellImage::GetWorkingMemorySize(memsize, imageRowstride/2, imageHeight/2,
										CELL_IMAGE_DATA_ALLOCATE_EXTERNAL);
	memsize += CellImage::GetWorkingMemorySize(memsize, imageRowstride/4, imageHeight/4,
										CELL_IMAGE_DATA_ALLOCATE_EXTERNAL);
	
	memsize += sizeof_byte_align(16, memsize, sizeof(CellImage *) * maxTrackingNum);
	memsize += sizeof_byte_align(4, memsize, sizeof(int) * maxTrackingNum);
	memsize += sizeof_byte_align(4, memsize, sizeof(int) * maxTrackingNum);
	memsize += sizeof_byte_align(4, memsize, sizeof(int) * maxTrackingNum);
	memsize += sizeof_byte_align(4, memsize, sizeof(int) * maxTrackingNum);
	
	memsize += sizeof_byte_align(FaceRegisterInfo::GetAlign(), memsize,
								 sizeof(FaceRegisterInfo) * maxTrackingNum);
	for(int i=0; i<maxTrackingNum; i++) {
		memsize += FaceRegisterInfo::GetWorkingMemorySize(memsize);
	}
	
	memsize += sizeof_byte_align(LocalSearchInfo::GetAlign(), memsize,
								 sizeof(LocalSearchInfo) * maxTrackingNum);
	for(int i=0; i<maxTrackingNum; i++) {
		memsize += LocalSearchInfo::GetWorkingMemorySize(memsize);
	}
	
	memsize += sizeof_byte_align(TemplateMatchingInfo::GetAlign(), memsize,
								 sizeof(TemplateMatchingInfo) * maxTrackingNum);
	for(int i=0; i<maxTrackingNum; i++) {
		memsize += TemplateMatchingInfo::GetWorkingMemorySize(memsize);
	}

	memsize += sizeof_byte_align(4, memsize, sizeof(int) * maxTrackingNum);
	
	memsize += GlobalFaceSearch::GetWorkingMemorySize(
		memsize, imageWidth, imageHeight, imageRowstride, maxTrackingNum);

	memsize += LocalFaceSearch::GetWorkingMemorySize(
		memsize, imageWidth, imageHeight, imageRowstride, maxTrackingNum);

	memsize += TemplateMatching::GetWorkingMemorySize(memsize);

	memsize += sizeof_byte_align(CELL_FACE_TRACKER_RESULT_ALIGN, memsize,
								 sizeof(CellFaceTrackerResult) * maxTrackingNum);
	
	*worksize = memsize;
	
	return CELL_FACE_TRACKER_OK;
}

/////////////////////////////////////////////////////
// cellFaceTrackerInitialize
// return values : error code
/////////////////////////////////////////////////////
int cellFaceTrackerInitialize(
	const void *workingArea,
	int worksize,
	const CellSpurs2 *spurs,
	int imageWidth,
	int imageHeight,
	int imageRowstride,
	int maxTrackingNum,
	int faceDetectionInterval
	)
{
	if(gCellFaceTrackerInitFlag) {
		return CELL_FACE_TRACKER_ERROR_ALREADY_INITIALIZED;
	}
	if(spurs == NULL) {
		return CELL_FACE_TRACKER_ERROR_INVALID_PARAM;
	}
	if(workingArea == NULL) {
		return CELL_FACE_TRACKER_ERROR_INVALID_WORKAREA;
	}
	if(((unsigned int)workingArea % CELL_FACE_TRACKER_ALIGN) != 0) { //align check
		return CELL_FACE_TRACKER_ERROR_WORKAREA_ALIGN;
	}
	if((imageRowstride < 0) ||
	   (imageRowstride > CELL_FACE_TRACKER_IMG_WIDTH_MAX)) {//imageRowstride
		return CELL_FACE_TRACKER_ERROR_IMAGE_WIDTH_LIMIT;
	}
	if((imageRowstride&0x0000000f) != 0) { //imageRowstride 16Byte align check
		return CELL_FACE_TRACKER_ERROR_ROWSTRIDE_ALIGN;
	}
	if((imageWidth < 0) ||
	   (imageWidth > CELL_FACE_TRACKER_IMG_WIDTH_MAX)) { //imageWidth check
		return CELL_FACE_TRACKER_ERROR_IMAGE_WIDTH_LIMIT;
	}
	if(imageHeight < 0 ||
	   imageHeight > CELL_FACE_TRACKER_IMG_HEIGHT_MAX ||
	   maxTrackingNum < 0 ||
	   maxTrackingNum > CELL_FACE_DETECT_NUM_MAX) {
		return CELL_FACE_TRACKER_ERROR_INVALID_PARAM;
	}
	if(faceDetectionInterval <= 0 ||
	   faceDetectionInterval > CELL_FACE_TRACKER_DETECTION_INTERVAL_MAX) {
		return CELL_FACE_TRACKER_ERROR_INVALID_PARAM;
	}

	/////////////////////////////////////////////////////
	// memory assign
	/////////////////////////////////////////////////////
	char *p = (char *)workingArea;
	p += assign_memory_byte_align(CELL_FACE_TRACKER_ALIGN, (void *)p, (void *)&gFT,
								  sizeof(CellFaceTracker));
	new(gFT) CellFaceTracker;
	
	p += assign_memory_byte_align(CELL_IMAGE_DATA_ALIGN, (void *)p, (void *)&gFT->mImageQVGA,
								  sizeof(unsigned char) * (imageRowstride/2) * (imageHeight/2));
	p += assign_memory_byte_align(CELL_IMAGE_DATA_ALIGN, (void *)p, (void *)&gFT->mImageQQVGA,
								  sizeof(unsigned char) * (imageRowstride/4) * (imageHeight/4));

	p += gFT->mCellImageVGA.Initialize(p, imageRowstride, imageHeight,
									   CELL_IMAGE_DATA_ALLOCATE_EXTERNAL);
	p += gFT->mCellImageQVGA.Initialize(p, imageRowstride/2, imageHeight/2,
										CELL_IMAGE_DATA_ALLOCATE_EXTERNAL);
	p += gFT->mCellImageQQVGA.Initialize(p, imageRowstride/4, imageHeight/4,
										 CELL_IMAGE_DATA_ALLOCATE_EXTERNAL);

	
	p += assign_memory_byte_align(16, (void *)p, (void *)&gFT->mCellLayerImage,
								  sizeof(CellImage *) * maxTrackingNum);
	
	p += assign_memory_byte_align(4, (void *)p, (void *)&gFT->mTargetStatus,
								  sizeof(int) * maxTrackingNum);
	p += assign_memory_byte_align(4, (void *)p, (void *)&gFT->mTrackCount,
								  sizeof(int) * maxTrackingNum);
	p += assign_memory_byte_align(4, (void *)p, (void *)&gFT->mLocalSearchTimingOffset,
								  sizeof(int) * maxTrackingNum);
	p += assign_memory_byte_align(4, (void *)p, (void *)&gFT->mTemplateLayer,
								  sizeof(int) * maxTrackingNum);

	
	p += assign_memory_byte_align(FaceRegisterInfo::GetAlign(), (void *)p,
								  (void *)&gFT->mFaceRegisterInfo,
								  sizeof(FaceRegisterInfo) * maxTrackingNum);

	for(int i=0; i<maxTrackingNum; i++) {
		new(&gFT->mFaceRegisterInfo[i]) FaceRegisterInfo;
		p += gFT->mFaceRegisterInfo[i].Initialize(p);
	}
	p += assign_memory_byte_align(LocalSearchInfo::GetAlign(), (void *)p,
								  (void *)&gFT->mLocalSearchInfo,
								  sizeof(LocalSearchInfo) * maxTrackingNum);
	for(int i=0; i<maxTrackingNum; i++) {
		new(&gFT->mLocalSearchInfo[i]) LocalSearchInfo;
		p += gFT->mLocalSearchInfo[i].Initialize(p);
	}
	p += assign_memory_byte_align(TemplateMatchingInfo::GetAlign(), (void *)p,
								  (void *)&gFT->mTemplateMatchingInfo,
								  sizeof(TemplateMatchingInfo) * maxTrackingNum);
	for(int i=0; i<maxTrackingNum; i++) {
		new(&gFT->mTemplateMatchingInfo[i]) TemplateMatchingInfo;
		p += gFT->mTemplateMatchingInfo[i].Initialize(p);
	}

	p += assign_memory_byte_align(4, (void *)p, (void *)&gFT->mUsedTemplate,
								  sizeof(int) * maxTrackingNum);

	p += gFT->mGlobalSearch.Initialize(p, spurs, imageWidth, imageHeight, imageRowstride, maxTrackingNum);
	p += gFT->mLocalSearch.Initialize(p, spurs, imageWidth, imageHeight, imageRowstride, maxTrackingNum);
	p += gFT->mTemplateMatching.Initialize(p, spurs);


	p += assign_memory_byte_align(CELL_FACE_TRACKER_RESULT_ALIGN, (void *)p,
								  (void *)&(gResult),
								  sizeof(CellFaceTrackerResult) * maxTrackingNum);

	////////////////////////////////////////
	// assign error check
	////////////////////////////////////////
	if(((int)p - (int)workingArea) > worksize) {
		gFT->mCellImageVGA.Finalize();
		gFT->mCellImageQVGA.Finalize();
		gFT->mCellImageQQVGA.Finalize();
		for(int i=0; i<maxTrackingNum; i++) {
			gFT->mFaceRegisterInfo[i].Finalize();
		}
		for(int i=0; i<maxTrackingNum; i++) {
			gFT->mLocalSearchInfo[i].Finalize();
		}
		for(int i=0; i<maxTrackingNum; i++) {
			gFT->mTemplateMatchingInfo[i].Finalize();
		}
		gFT->mGlobalSearch.Finalize();
		gFT->mLocalSearch.Finalize();
		gFT->mTemplateMatching.Finalize();
		
		return CELL_FACE_TRACKER_ERROR_SHORT_WORKSIZE;
	}

	////////////////////////////////////////
	// spurs taskset initialize
	////////////////////////////////////////
	const unsigned char priorities[8] = {1, 1, 1, 1, 1, 1, 1, 1};
	int ret = sampleVisionUtilInitializeEx((CellSpurs2*)spurs, &gFT->mSpursTaskset, priorities, 2);
	if (ret != CELL_OK) return ret;
	
	///////////////////////////////////
	// parameter initialize
	///////////////////////////////////
	gFT->mImageWidth = imageWidth;
	gFT->mImageHeight = imageHeight;
	gFT->mImageRowstride = imageRowstride;
	memset(gFT->mImageQVGA, 0, imageRowstride * imageHeight / 4);
	memset(gFT->mImageQQVGA, 0, imageRowstride * imageHeight / 16);

	gFT->mCellImageQVGA.SetData(gFT->mImageQVGA);
	gFT->mCellImageQQVGA.SetData(gFT->mImageQQVGA);

	gFT->mProcessCount = 0;
	
	gFT->mMaxTarget = maxTrackingNum;

	for(int i=0; i<maxTrackingNum; i++) {
		gFT->mTargetStatus[i] = TARGET_STATUS_NULL;
		gFT->mTrackCount[i] = 0;
		gFT->mLocalSearchTimingOffset[i] = 0;
		gFT->mTemplateLayer[i] = TEMPLATE_LAYER_VGA;
		gFT->mUsedTemplate[i] = TM_TEMPLATE_NUM_MAX;
		gFT->mCellLayerImage[i] = &gFT->mCellImageVGA;
	}

	gFT->mLocalSearchInterval = faceDetectionInterval;

	for(int i=0; i<maxTrackingNum; i++) {
		gResult[i].status = CELL_FACE_TRACKER_STATUS_NOTTRACKING;
		gResult[i].id = 0;
		gResult[i].posX = 0.0f;
		gResult[i].posY = 0.0f;
		gResult[i].score = 0.0f;
	}

	gRecogTimeTotal = 0ULL;
	gTrackingTimeTotal = 0ULL;
	gTimerStart = 0ULL;
	gTimerEnd = 0ULL;
	gCellFaceTrackerCallbackFunc = NULL;
	
	gCellFaceTrackerInitFlag = true;

	return CELL_FACE_TRACKER_OK;
}

/////////////////////////////////////////////////////
// cellFaceTrackerInitialize
// return values : ok/ng flag
/////////////////////////////////////////////////////
int cellFaceTrackerFinalize(void)
{
	int ret;
	
	if(!gCellFaceTrackerInitFlag) {
		return CELL_FACE_TRACKER_ERROR_NOT_INITIALIZED;
	}

	gFT->mCellImageVGA.Finalize();
	gFT->mCellImageQVGA.Finalize();
	gFT->mCellImageQQVGA.Finalize();
	
	for(int i=0; i<gFT->mMaxTarget; i++) {
		gFT->mFaceRegisterInfo[i].Finalize();
	}
	for(int i=0; i<gFT->mMaxTarget; i++) {
		gFT->mLocalSearchInfo[i].Finalize();
	}
	for(int i=0; i<gFT->mMaxTarget; i++) {
		gFT->mTemplateMatchingInfo[i].Finalize();
	}
	gFT->mGlobalSearch.Finalize();
	gFT->mLocalSearch.Finalize();
	gFT->mTemplateMatching.Finalize();
	ret = sampleVisionUtilFinalizeEx(&gFT->mSpursTaskset);
	if (ret != CELL_OK) return ret;
	
	gCellFaceTrackerInitFlag = false;
	
	return CELL_FACE_TRACKER_OK;
}

/////////////////////////////////////////
// cellFaceTrackerUpdate
// return values : result flag
/////////////////////////////////////////
int cellFaceTrackerUpdate(const unsigned char *imageData)
{
	if(!gCellFaceTrackerInitFlag) {
		return CELL_FACE_TRACKER_ERROR_NOT_INITIALIZED;
	}
	if(imageData == NULL)
	{
		return CELL_FACE_TRACKER_ERROR_INVALID_PARAM;
	}

	SYS_TIMEBASE_GET(gTimerStart); //gRecogTimeTotal

	/////////////////////////////////////////
	// make layer images
	/////////////////////////////////////////
	//VGA
	gFT->mCellImageVGA.SetData((unsigned char *)imageData);
	//VGA -> QVGA
	sampleVisionUtilResizeHalf(&gFT->mSpursTaskset, &gFT->mResizeHalfTask,
							   &gFT->mCellImageVGA, &gFT->mCellImageQVGA);
	//QVGA -> QQVGA
	sampleVisionUtilResizeHalf(&gFT->mSpursTaskset, &gFT->mResizeHalfTask,
							   &gFT->mCellImageQVGA, &gFT->mCellImageQQVGA);
	
	///////////////////////////////////////////////////////////////
	// global search
	///////////////////////////////////////////////////////////////
	int numFace = 0;
	
	int ret = gFT->mGlobalSearch.ExecFaceSearch(
		imageData,
		(const int *)gFT->mTargetStatus,
		(const FaceRegisterInfo *)gFT->mFaceRegisterInfo,
		(const LocalSearchInfo *)gFT->mLocalSearchInfo,
		(const CellFaceTrackerCallbackFunc)gCellFaceTrackerCallbackFunc,
		&numFace);
	
	if(ret == GLOBAL_FACE_SEARCH_OK) { //OK
		for(int num=0; num<numFace; num++) { //numFace loop
			//set ready
			int tgt = gFT->mGlobalSearch.GetRegisteredTargetNum(num);
			if(gFT->mTargetStatus[tgt] == TARGET_STATUS_EMPTY) {
				gFT->mLocalSearchInfo[tgt].SetDetectResult(
					gFT->mGlobalSearch.GetDetectResult(num));
				gFT->mTargetStatus[tgt] = TARGET_STATUS_READY; // ⇒READY
			} //EMPTY
		} //numFace loop
	} //OK
	
	///////////////////////////////////////////////////////////////
	// local search and create template for template matching
	///////////////////////////////////////////////////////////////
	for(int tgt=0; tgt<gFT->mMaxTarget; tgt++) { //target loop
		//LS timing check
		int status = gFT->mTargetStatus[tgt];
		int intTiming = gFT->mProcessCount % gFT->mLocalSearchInterval;
		
		if(status == TARGET_STATUS_READY || //ready
		   ((status == TARGET_STATUS_TRACKING) &&
			(intTiming == gFT->mLocalSearchTimingOffset[tgt])) || //interval timing
		   status == TARGET_STATUS_LOST) { //retry
			
			/////////////////////////////////////////////////
			// Local Search
			/////////////////////////////////////////////////
			ret = gFT->mLocalSearch.ExecFaceSearch(
				imageData,
				(const int *)gFT->mTargetStatus,
				(const LocalSearchInfo *)gFT->mLocalSearchInfo,
				tgt);
			
			if((ret == LOCAL_FACE_SEARCH_OK) &&
			   (gFT->mLocalSearch.GetOverlapFlag() == false)) { //detect not overlaped face

				//get face detection result
				const CellFaceDetectionResult *face = gFT->mLocalSearch.GetCurrResult();
				const CellFacePartsResult *parts = gFT->mLocalSearch.GetPartsResultAll();

				if(gFT->mLocalSearch.GetDetectPartsFlag()) { //4 points parts detected
					if(status == TARGET_STATUS_READY) {
						//update local search position and lost count
						cellFaceTrackerUpdateDataAfterLocalSearch(tgt, face);
						
						gFT->mTrackCount[tgt] = 0;
						//set local search timing
						gFT->mLocalSearchTimingOffset[tgt] =
							(int)(tgt * gFT->mLocalSearchInterval / gFT->mMaxTarget);
						
						//calc layer
						int candLayer = cellFaceTrackerSelectTemplateLayer(
							face, parts, gFT->mImageWidth, gFT->mImageHeight);

						//set layer image and tempate
						cellFaceTrackerUpdateDataBeforeTemplateMatching(tgt, candLayer);
						cellFaceTrackerSetLayerImageAndTemplate(tgt, candLayer);
						
						//change status from READY to TRACKING
						gFT->mTargetStatus[tgt] = TARGET_STATUS_TRACKING; //⇒TRACKING
					} //TARGET_STATUS_READY
					else if(status == TARGET_STATUS_TRACKING) {
						//update local search position and lost count
						cellFaceTrackerUpdateDataAfterLocalSearch(tgt, face);
						
						//calc layer
						int prevLayer = gFT->mTemplateLayer[tgt];
						int candLayer = cellFaceTrackerSelectTemplateLayer(
							face, parts, gFT->mImageWidth, gFT->mImageHeight);
						
						//check all template update timing
						if(prevLayer != candLayer) { //template update timing
							//set layer image and tempate
							cellFaceTrackerUpdateDataBeforeTemplateMatching(tgt, candLayer);
							cellFaceTrackerSetLayerImageAndTemplate(tgt, candLayer);
						}
						else { //not template updata timing
							//set tempate matching position
							cellFaceTrackerUpdateDataBeforeTemplateMatching(tgt, prevLayer);
						}
					} //TARGET_STATUS_TRACKING
					else if(status == TARGET_STATUS_LOST) {
						//update local search position and lost count
						cellFaceTrackerUpdateDataAfterLocalSearch(tgt, face);
						
						//calc layer
						int prevLayer = gFT->mTemplateLayer[tgt];
						int candLayer = cellFaceTrackerSelectTemplateLayer(
							face, parts, gFT->mImageWidth, gFT->mImageHeight);
						
						//check all template update timing
						if(prevLayer != candLayer) { //template update timing
							//set layer image and tempate
							cellFaceTrackerUpdateDataBeforeTemplateMatching(tgt, candLayer);
							cellFaceTrackerSetLayerImageAndTemplate(tgt, candLayer);
						}
						else { //not template updata timing
							//set tempate matching position
							cellFaceTrackerUpdateDataBeforeTemplateMatching(tgt, prevLayer);
						} //not template updata timing
						
						//change status from LOST to TRACKING
						gFT->mTargetStatus[tgt] = TARGET_STATUS_TRACKING; //⇒TRACKING
					} //TARGET_STATUS_LOST
				} //detect 4 parts
				else { //not detect 4 parts
					if(status == TARGET_STATUS_READY) {
						gFT->mTargetStatus[tgt] = TARGET_STATUS_EMPTY; //⇒EMPTY
					}
					else if(status == TARGET_STATUS_TRACKING){ //TRACKING or LOST
						//update local search position and lost count
						cellFaceTrackerUpdateDataAfterLocalSearch(tgt, face);
					}
					else if(status == TARGET_STATUS_LOST){ //LOST
						//update local search position and lost count
						cellFaceTrackerUpdateDataAfterLocalSearch(tgt, face);
						//change status from LOST to TRACKING
						gFT->mTargetStatus[tgt] = TARGET_STATUS_TRACKING; //⇒TRACKING
					}
				} //not detect 4 parts
			} //detect not overlaped face
			else { //not detect face  or  detect overlap face
				if(status == TARGET_STATUS_READY) {
					//change status from READY to EMPTY
					gFT->mTargetStatus[tgt] = TARGET_STATUS_EMPTY; //⇒EMPTY
				}
				else if(status == TARGET_STATUS_TRACKING) {
					gFT->mLocalSearchInfo[tgt].IncrementLostCount();
					//change status from TRACKING to LOST
					gFT->mTargetStatus[tgt] = TARGET_STATUS_LOST; //⇒LOST
				}
				else if(status == TARGET_STATUS_LOST) {
					gFT->mLocalSearchInfo[tgt].IncrementLostCount();
				}
			} //not detect face  or  detect overlap face
		} //READY / LS timing / retry
	} //target loop

	SYS_TIMEBASE_GET(gTimerEnd); //gRecogTimeTotal
	gRecogTimeTotal = gTimerEnd - gTimerStart;

	SYS_TIMEBASE_GET(gTimerStart); //gTrackingTimeTotal
	
	///////////////////////////////////////////////////////////////
	// template matching
	///////////////////////////////////////////////////////////////
	for(int tgt=0; tgt<gFT->mMaxTarget; tgt++) { //target loop
		
		int status = gFT->mTargetStatus[tgt];
		
		if(status == TARGET_STATUS_TRACKING ||
		   status == TARGET_STATUS_LOST) { //valid status
			
			ret = gFT->mTemplateMatching.ExecTemplateMatching(
				gFT->mCellLayerImage[tgt], &gFT->mTemplateMatchingInfo[tgt]);
			gFT->mUsedTemplate[tgt] = gFT->mTemplateMatching.GetUsedTemplate();
			
			if(ret == TEMPLATE_MATCHING_OK) { //match
				if(gFT->mUsedTemplate[tgt] <= TM_TEMPLATE_ADDITIONAL) { //match additional or previous TP
					//update template matching result and local search position
					cellFaceTrackerUpdateDataAfterTemplateMatching(tgt);
					//update previous template
					cellFaceTrackerUpdatePreviousTemplate(tgt);
					
				} //match additional or previous TP
				else if(gFT->mUsedTemplate[tgt] == TM_TEMPLATE_PREVIOUS) {

					//update template matching result and local search position
					cellFaceTrackerUpdateDataAfterTemplateMatching(tgt);
					//update additional and previous template
					cellFaceTrackerUpdateAdditionalAndPreviousTemplate(tgt);
					
				}  //match previous TP
			} //match
			else { //not match or error
				if(status == TARGET_STATUS_TRACKING) {
					gFT->mTemplateMatchingInfo[tgt].IncrementLostCount();
					//change status from TRACKING to LOST
					gFT->mTargetStatus[tgt] = TARGET_STATUS_LOST; //⇒LOST
				}
				else if(status == TARGET_STATUS_LOST) {
					gFT->mTemplateMatchingInfo[tgt].IncrementLostCount();
				}
			} //not match or error
		} //valid status
	} //target loop
	
	SYS_TIMEBASE_GET(gTimerEnd); //gTrackingTimeTotal
	gTrackingTimeTotal = gTimerEnd - gTimerStart;
	
	///////////////////////////////////////////////////////////////
	// post process
	///////////////////////////////////////////////////////////////
	for(int tgt=0; tgt<gFT->mMaxTarget; tgt++) { //target loop

		CellFaceTrackerResult *result = &gResult[tgt];
		int status = gFT->mTargetStatus[tgt];

		////////////////////////////////////
		//update parameter and set result
		////////////////////////////////////
		if(status == TARGET_STATUS_NULL ||
		   status == TARGET_STATUS_EMPTY ||
		   status == TARGET_STATUS_READY) {
			//set result
			result->status = CELL_FACE_TRACKER_STATUS_NOTTRACKING;
			result->id = 0;
			result->posX = 0.0f;
			result->posY = 0.0f;
			result->score = 0.0f; //tentative
		}
		else if(status == TARGET_STATUS_TRACKING) { //TRACKING
			// update parameter
			gFT->mTrackCount[tgt]++;
			gFT->mTemplateMatchingInfo[tgt].ClearLostCount();
			
			// calc template center
			float posx;
			float posy;
			gFT->mTemplateMatchingInfo[tgt].GetPosition(&posx, &posy);
			int layer = gFT->mTemplateLayer[tgt];
			cellFaceTrackerConvOriginalLayerPosition(layer, &posx, &posy);

			//set result
			result->status = CELL_FACE_TRACKER_STATUS_TRACKING;
			result->id = gFT->mFaceRegisterInfo[tgt].GetID();
			result->posX = posx;
			result->posY = posy;
			result->score = 0.0f; //tentative

		} //TRACKING
		else if(status == TARGET_STATUS_LOST) { //LOST
			//judge empty
			if(gFT->mLocalSearchInfo[tgt].GetLostCount() > CELL_FACE_TRACKER_LS_EMPTY_THRE ||
			   gFT->mTemplateMatchingInfo[tgt].GetLostCount() > CELL_FACE_TRACKER_TM_EMPTY_THRE) {
				//change status from LOST to EMPTY
				gFT->mTargetStatus[tgt] = TARGET_STATUS_EMPTY; //⇒EMPTY
				// clear parameter
				gFT->mTrackCount[tgt] = 0;
				gFT->mLocalSearchInfo[tgt].ClearLostCount();
				gFT->mTemplateMatchingInfo[tgt].ClearLostCount();
				gFT->mLocalSearchTimingOffset[tgt] = 0;
				gFT->mTemplateMatchingInfo[tgt].ClearTemplateAll();
				
				//set result
				result->status = CELL_FACE_TRACKER_STATUS_NOTTRACKING;
				result->id = 0;
				result->posX = 0.0f;
				result->posY = 0.0f;
				result->score = 0.0f; //tentative
				
			} //LOST ⇒ EMPTY
			else { //LOST
				gFT->mTrackCount[tgt]++;
				// calc template center
				float posx;
				float posy;
				gFT->mTemplateMatchingInfo[tgt].GetPosition(&posx, &posy);
				int layer = gFT->mTemplateLayer[tgt];
				cellFaceTrackerConvOriginalLayerPosition(layer, &posx, &posy);
				
				//set result
				result->status = CELL_FACE_TRACKER_STATUS_LOST;
				result->id = gFT->mFaceRegisterInfo[tgt].GetID();
				result->posX = posx;
				result->posY = posy;
				result->score = 0.0f; //tentative
			} //LOST
		}
	} //target loop

	gFT->mProcessCount++;

	
#if VERVOSE_DEBUG
	fprintf(stderr, "# Target Information [%d] (after post process)\n",
			gFT->mProcessCount);
	for(int i=0; i<gFT->mMaxTarget; i++) {
		cellFaceTrackerPrintTargetInfo(stderr, i);
	}
#endif //VERVOSE_DEBUG

	return CELL_FACE_TRACKER_OK;
}


////////////////////////////////////////
// cellFaceTrackerGetResult
////////////////////////////////////////
int cellFaceTrackerGetResult(int id, CellFaceTrackerResult *result)
{
	if(!gCellFaceTrackerInitFlag) {
		return CELL_FACE_TRACKER_ERROR_NOT_INITIALIZED;
	}
	
	if(result == NULL) {
		return CELL_FACE_TRACKER_ERROR_INVALID_PARAM;
	}
	
	if(id < 0) {
		return CELL_FACE_TRACKER_ERROR_INVALID_PARAM;
	}
	
	for(int i=0; i<gFT->mMaxTarget; i++) {
		if(gFT->mFaceRegisterInfo[i].GetID() == id) {
			result->id     = gResult[i].id;
			result->posX   = gResult[i].posX;
			result->posY   = gResult[i].posY;
			result->score  = gResult[i].score;
			result->status = gResult[i].status;

			return CELL_FACE_TRACKER_OK;
		}
	}

	return CELL_FACE_TRACKER_ERROR_FACEID_NOT_EXIST;
}

//////////////////////////////////////////////////////////////////////////////
// private function
//////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////
// cellFaceTrackerCalcFaceDetectResultCenter
////////////////////////////////////////
int cellFaceTrackerCalcFaceDetectResultCenter(
	const CellFaceDetectionResult *result,
	float *xc, float *yc)
{

	float x = result->faceX;
	float y = result->faceY;
	float w = result->faceW;
	float h = result->faceH;

	*xc = (x + w / 2.0f) * gFT->mImageWidth;
	*yc = (y + h / 2.0f) * gFT->mImageHeight;
	
	return CELL_FACE_TRACKER_OK;
}


////////////////////////////////////////
// cellFaceTrackerSelectTemplateLayer
////////////////////////////////////////
int cellFaceTrackerSelectTemplateLayer(const CellFaceDetectionResult *face,
									   const CellFacePartsResult *parts,
									   int width, int height)
{
#define SIZE_THRE_VGA_QVGA (24.0f)
#define SIZE_THRE_QVGA_QQVGA (48.0f)
	
	float faceW = face->faceW;
	float faceH = face->faceH;
	
	float partsRX = parts[CELL_FACE_PARTS_ID_R_EYE_CENTER - 1].partsX;
	float partsRY = parts[CELL_FACE_PARTS_ID_R_EYE_CENTER - 1].partsY;
	float partsLX = parts[CELL_FACE_PARTS_ID_L_EYE_CENTER - 1].partsX;
	float partsLY = parts[CELL_FACE_PARTS_ID_L_EYE_CENTER - 1].partsY;
	
	float distX = (partsRX - partsLX) * faceW*width;
	float distY = (partsRY - partsLY) * faceH*height;
	float dist2 = distX * distX + distY * distY;

	//select layer
	if(dist2 <= SIZE_THRE_VGA_QVGA * SIZE_THRE_VGA_QVGA) {
		return TEMPLATE_LAYER_VGA;
	} else if(dist2 <= SIZE_THRE_QVGA_QQVGA * SIZE_THRE_QVGA_QQVGA) {
		return TEMPLATE_LAYER_QVGA;
	} else {
		return TEMPLATE_LAYER_QQVGA;
	}
}


////////////////////////////////////////
// cellFaceTrackerSetLayerImage
////////////////////////////////////////
void cellFaceTrackerSetLayerImage(int tgt, int layer)
{
	if(layer == TEMPLATE_LAYER_VGA) {
		gFT->mCellLayerImage[tgt] = &gFT->mCellImageVGA;
	}
	else if(layer == TEMPLATE_LAYER_QVGA) {
		gFT->mCellLayerImage[tgt] = &gFT->mCellImageQVGA;
	}
	else if(layer == TEMPLATE_LAYER_QQVGA) {
		gFT->mCellLayerImage[tgt] = &gFT->mCellImageQQVGA;
	}
	else {
		gFT->mCellLayerImage[tgt] = &gFT->mCellImageVGA;
	}
}

////////////////////////////////////////
// cellFaceTrackerConvLayerPosition
////////////////////////////////////////
void cellFaceTrackerConvLayerPosition(int layer, float *x, float *y)
{
	if(layer == TEMPLATE_LAYER_VGA) {
	}
	else if(layer == TEMPLATE_LAYER_QVGA) {
		*x /= 2.0f;
		*y /= 2.0f;
	}
	else if(layer == TEMPLATE_LAYER_QQVGA) {
		*x /= 4.0f;
		*y /= 4.0f;
	}
}


////////////////////////////////////////
// cellFaceTrackerConvOriginalLayerPosition
////////////////////////////////////////
void cellFaceTrackerConvOriginalLayerPosition(int layer, float *x, float *y)
{
	if(layer == TEMPLATE_LAYER_VGA) {
	}
	else if(layer == TEMPLATE_LAYER_QVGA) {
		*x *= 2.0f;
		*y *= 2.0f;
	}
	else if(layer == TEMPLATE_LAYER_QQVGA) {
		*x *= 4.0f;
		*y *= 4.0f;
	}
}


////////////////////////////////////////
// cellFaceTrackerUpdateDataAfterLocalSearch
////////////////////////////////////////
void cellFaceTrackerUpdateDataAfterLocalSearch(
	int tgt,
	const CellFaceDetectionResult *face
	)
{
	//set local search position
	gFT->mLocalSearchInfo[tgt].SetDetectResult(face);
	gFT->mLocalSearchInfo[tgt].ClearLostCount();
	gFT->mTemplateMatchingInfo[tgt].ClearLostCount();
}


////////////////////////////////////////
// cellFaceTrackerUpdateDataBeforeTemplateMatching
////////////////////////////////////////
void cellFaceTrackerUpdateDataBeforeTemplateMatching(int tgt, int layer)
{
	float posx;
	float posy;
	
	gFT->mLocalSearch.GetPositionPartsEyeCenter(&posx, &posy);
	//set tm position
	cellFaceTrackerConvLayerPosition(layer, &posx, &posy);
	gFT->mTemplateMatchingInfo[tgt].SetPosition(posx, posy);
}


////////////////////////////////////////
// cellFaceTrackerSetLayerImageAndTemplate
////////////////////////////////////////
void cellFaceTrackerSetLayerImageAndTemplate(int tgt, int layer)
{
	float posx;
	float posy;
	float yaw;
	float pitch;
	float roll;
	
	gFT->mTemplateMatchingInfo[tgt].GetPosition(&posx, &posy);
		
	//set tm layer image
	cellFaceTrackerSetLayerImage(tgt, layer);
	
	//update all template of new position
	gFT->mLocalSearch.GetPositionResultPose(&yaw, &pitch, &roll);
	gFT->mTemplateMatchingInfo[tgt].CreateTemplateAll
		(gFT->mCellLayerImage[tgt]->GetData(),
		 gFT->mCellLayerImage[tgt]->GetWidth(),
		 gFT->mCellLayerImage[tgt]->GetHeight(),
		 posx, posy, roll);

	//set tm layer
	gFT->mTemplateLayer[tgt] = layer;
}

////////////////////////////////////////
// cellFaceTrackerUpdateDataAfterTemplateMatching
////////////////////////////////////////
void cellFaceTrackerUpdateDataAfterTemplateMatching(int tgt)
{
	const TemplateMatchingResult *tmResult = gFT->mTemplateMatching.GetResult();

	float inx;
	float iny;
	float outx;
	float outy;
	
	gFT->mTemplateMatchingInfo[tgt].GetPosition(&inx, &iny); //input position
	outx = tmResult->mPosX;
	outy = tmResult->mPosY;
	
	//update template matching info
	gFT->mTemplateMatchingInfo[tgt].SetPosition(outx, outy);
	gFT->mTemplateMatchingInfo[tgt].ClearLostCount();
	
	//update local search info
	gFT->mLocalSearchInfo[tgt].SetDetectResultOffset(
		(outx - inx) / gFT->mCellLayerImage[tgt]->GetWidth(),
		(outy - iny) / gFT->mCellLayerImage[tgt]->GetHeight());
}


////////////////////////////////////////
// cellFaceTrackerUpdatePreviousTemplate
////////////////////////////////////////
void cellFaceTrackerUpdatePreviousTemplate(int tgt)
{
	float outx;
	float outy;
	
	gFT->mTemplateMatchingInfo[tgt].GetPosition(&outx, &outy);
	gFT->mTemplateMatchingInfo[tgt].UpdateTemplatePrev(
		gFT->mCellLayerImage[tgt]->GetData(),
		gFT->mCellLayerImage[tgt]->GetWidth(),
		gFT->mCellLayerImage[tgt]->GetHeight(),
		outx, outy);
}

////////////////////////////////////////
// cellFaceTrackerUpdateAdditionalAndPreviousTemplate
////////////////////////////////////////
void cellFaceTrackerUpdateAdditionalAndPreviousTemplate(int tgt)
{
	//update additional tp (before update prev tp)
	gFT->mTemplateMatchingInfo[tgt].UpdateTemplateAdditional();
	//update previous template
	cellFaceTrackerUpdatePreviousTemplate(tgt);
}

/////////////////////////////////////////////////////
// for DEBUG
/////////////////////////////////////////////////////

////////////////////////////////////////
// cellFaceTrackerPrintTargetInfo
////////////////////////////////////////
void cellFaceTrackerPrintTargetInfo(FILE *fp, int tgt)
{
	CellFaceDetectionResult const *result = gFT->mLocalSearchInfo[tgt].GetDetectResult();
	float tx;
	float ty;
	gFT->mTemplateMatchingInfo[tgt].GetPosition(&tx, &ty);

	fprintf(fp, "#[%d]\t%d\t%d\t%d\t%d\t%d\t%d\t%d\t%f\t%f\t%f\t%f\t%f\t%f\t%f\t%f\t%f\t%d\t%d\t%d\t\n",
			tgt,
			gFT->mTargetStatus[tgt],
			gFT->mTrackCount[tgt],
			gFT->mLocalSearchTimingOffset[tgt],
			gFT->mFaceRegisterInfo[tgt].GetID(),
			gFT->mLocalSearchInfo[tgt].GetLostCount(),
			gFT->mCellLayerImage[tgt]->GetWidth(),
			gFT->mCellLayerImage[tgt]->GetHeight(),
			result->faceX,
			result->faceY,
			result->faceW,
			result->faceH,
			result->faceRoll,
			result->facePitch,
			result->faceYaw,
			tx, ty,
			TM_TEMPLATE_SIZE,
			gFT->mTemplateLayer[tgt],
			gFT->mTemplateMatchingInfo[tgt].GetLostCount()
			);


}

