/* SCE CONFIDENTIAL
 * PlayStation(R)3 Programmer Tool Runtime Library 475.001
 * Copyright (C) 2011 Sony Computer Entertainment Inc.
 * All Rights Reserved.
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <errno.h>
#include <cell/face.h>

#include <new>

#include "memory.h"
#include "TemplateMatchingInfo.h"


#define max(a, b) (((a) > (b)) ? (a) : (b))


////////////////////////////////////////////////
// Constructor
////////////////////////////////////////////////
TemplateMatchingInfo::
TemplateMatchingInfo()
{
	mInit = false;
}

////////////////////////////////////////////////
// Destructor
////////////////////////////////////////////////
TemplateMatchingInfo::
~TemplateMatchingInfo()
{
}

////////////////////////////////////////////////
// GetAlign
////////////////////////////////////////////////
int TemplateMatchingInfo::
GetAlign(void)
{
	int align = TEMPLATE_MATCHING_INFO_ALIGN;

	align = max(align, 16); //compare mTemplate[] align
	
	return align;
}

////////////////////////////////////////////////
// GetWorkingMemorySize
////////////////////////////////////////////////
int TemplateMatchingInfo::
GetWorkingMemorySize(int size)
{
	int memsize = size;

	return (memsize - size);
}

////////////////////////////////////////////////
// Initialize
////////////////////////////////////////////////
int TemplateMatchingInfo::
Initialize(const void *work)
{
	if (mInit) return TEMPLATE_MATCHING_INFO_ERROR;

	unsigned char *p = (unsigned char*)work;


	////////////////////////////////
	// set paramter
	////////////////////////////////
	mPosX = 0.0f;
	mPosY = 0.0f;
	memset(mTemplate, 0, TM_TEMPLATE_MEM_SIZE_ALL);
	mLostCount = 0;
	
	//set initialize flag
	mInit = true;
	
	return (int)(p - (unsigned char*)work);
}

////////////////////////////////////////////////
// Finalize
////////////////////////////////////////////////
int TemplateMatchingInfo::
Finalize(void)
{
	if (!mInit) return TEMPLATE_MATCHING_INFO_ERROR;

	mInit = false;
	
	return TEMPLATE_MATCHING_INFO_OK;
}

////////////////////////////////////////////////
// ClearTemplateAll
////////////////////////////////////////////////
void TemplateMatchingInfo::
ClearTemplateAll(void)
{
	memset(mTemplate, 0, TM_TEMPLATE_MEM_SIZE_ALL);
}

////////////////////////////////////////////////
// GetPosition
////////////////////////////////////////////////
void TemplateMatchingInfo::
GetPosition(float *x, float *y) const
{
	*x = mPosX;
	*y = mPosY;
}

////////////////////////////////////////////////
// SetPosition
////////////////////////////////////////////////
void TemplateMatchingInfo::
SetPosition(float x, float y)
{
	mPosX = x;
	mPosY = y;
}

////////////////////////////////////////////////
// SetTemplate
////////////////////////////////////////////////
void TemplateMatchingInfo::
SetTemplate(TM_TEMPLATE_NUM num, const unsigned char *tplate)
{
	memcpy(&mTemplate[num * TM_TEMPLATE_MEM_SIZE], tplate, TM_TEMPLATE_MEM_SIZE);
}



////////////////////////////////////////////////
// CreateTemplateAll
////////////////////////////////////////////////
void TemplateMatchingInfo::
CreateTemplateAll(const unsigned char *image, int width, int height, float x, float y, float roll)
{
	CropRollTemplate(TM_TEMPLATE_FRONT, image, width, height, width, x, y, roll);
	CropRollTemplate(TM_TEMPLATE_RIGHT_ROLL, image, width, height, width, x, y,
					 roll + 20.0f * M_PI / 180.0f);
	CropRollTemplate(TM_TEMPLATE_LEFT_ROLL, image, width, height, width, x, y,
					 roll - 20.0f * M_PI / 180.0f);

	SetTemplate(TM_TEMPLATE_ADDITIONAL, GetTemplate(TM_TEMPLATE_FRONT));
	SetTemplate(TM_TEMPLATE_PREVIOUS, GetTemplate(TM_TEMPLATE_FRONT));
}

////////////////////////////////////////////////
// UpdateTemplateAdditional
////////////////////////////////////////////////
void TemplateMatchingInfo::
UpdateTemplateAdditional(void)
{
	SetTemplate(TM_TEMPLATE_ADDITIONAL, GetTemplate(TM_TEMPLATE_PREVIOUS));
}

////////////////////////////////////////////////
// UpdateTemplatePrev
////////////////////////////////////////////////
void TemplateMatchingInfo::
UpdateTemplatePrev(const unsigned char *image, int width, int height, float x, float y)
{
	CropRollTemplate(TM_TEMPLATE_PREVIOUS, image, width, height, width, x, y, 0.0);
}


////////////////////////////////////////////////
// CropRollTemplate
// posx, posy : center position
// roll : detected object roll angle [radian] clockwise
// crop template with roll(clockwise) rotation in image
// x axis ->, y axis |
//                   v
////////////////////////////////////////////////
void TemplateMatchingInfo::
CropRollTemplate(TM_TEMPLATE_NUM num, const unsigned char *image,
				 int width, int height, int rowstride,
				 float posx, float posy, float roll)
{
	(void)width;
	(void)height;
	
	float cosr = cosf(roll);
	float sinr = sinf(roll);

	//counterclockswise rotation parameter
	// [ a b ]
	// [ c d ]
	float a = cosr;
	float b = sinr;
	float c = -sinr;
	float d = cosr;

	for(int x=0; x<TM_TEMPLATE_SIZE; x++) { // left to right
		for(int y=0; y<TM_TEMPLATE_SIZE; y++) { // up to down
			float xx = (float)x - (float)(TM_TEMPLATE_SIZE - 1) / 2.0f; // origin center shift
			float yy = (float)y - (float)(TM_TEMPLATE_SIZE - 1) / 2.0f; // origin center shift
			//rotation
			float xxr = a * xx + b * yy + posx;
			float yyr = c * xx + d * yy + posy;
			
			int xxri = (int)floorf(xxr);
			int yyri = (int)floorf(yyr);
			
			float dx = xxr - (float)xxri;
			float dy = yyr - (float)yyri;

			float Y00 = (float)image[rowstride * (yyri + 0) + (xxri + 0)];
			float Y10 = (float)image[rowstride * (yyri + 0) + (xxri + 1)];
			float Y01 = (float)image[rowstride * (yyri + 1) + (xxri + 0)];
			float Y11 = (float)image[rowstride * (yyri + 1) + (xxri + 1)];

			unsigned char *tplate = &mTemplate[TM_TEMPLATE_MEM_SIZE * num];
			tplate[y * TM_TEMPLATE_SIZE + x] =
				(unsigned char)((1.0 - dy) * ((1.0 - dx) * Y00 + dx * Y10)
					  + dy * ((1.0 - dx) * Y01 + dx * Y11) + 0.5);
		}
	}
}


