/* SCE CONFIDENTIAL
 * PlayStation(R)3 Programmer Tool Runtime Library 475.001
 *                Copyright (C) 2010 Sony Computer Entertainment Inc.
 *                                               All Rights Reserved.
 */

 
/*
 * GLU PSGL utility library
 */

#include <math.h>
#include <float.h>
#include <stdarg.h>		// For varargs
#include <stdio.h>
#include <string.h>

#include <PSGL/psgl.h>
#include <PSGL/psglu.h>
#include <PSGL/Types.h>
#include <PSGL/private.h>


//#define GLU_DEBUG	1


/** @addtogroup Utility
 *
 * @{
 */

/**
 * @short Set up a perspective projection matrix
 *
 * gluPerspectivef() specifies a viewing frustum into the world coordinate system.
 *
 * In general, the aspect ratio in gluPerspectivef() should match the aspect ratio of the associated viewport.
 * For example, aspect=2.0 means the viewer's angle of view is twice as wide in x as it is in y.
 * If the viewport is twice as wide as it is tall, it displays the image without distortion.
 * The matrix generated by gluPerspectivef() is multiplied by the current matrix, just as if glMultMatrixf() were called with the
 * generated matrix. To load the perspective matrix onto the current matrix stack instead, precede the call to
 * gluPerspectivef() with a call to glLoadIdentity().
 *
 * @param fovy    Specifies the field of view angle, in degrees, in the y direction
 * @param aspect  Specifies the aspect ratio that determines the field of view in the x direction.
 *                The aspect ratio is the ratio of x (width) to y (height)
 * @param znear   Specifies the distance from the viewer to the near clipping plane (always positive)
 * @param zfar    Specifies the distance from the viewer to the far clipping plane (always positive)
 *
 * @par Errors
Use glGetError() to retrieve the value of the error flag.
<TABLE>
<TR>
<TD><CODE>GL_INVALID_VALUE</CODE></TD>
<TD><I><c>znear</c></I> = <I><c>zfar</c></I>.</TD>
</TR>
</TABLE>
 */
void gluPerspectivef( GLfloat fovy, GLfloat aspect, GLfloat znear, GLfloat zfar )
{
    GLfloat	LXMin, LXMax, LYMin, LYMax;

    LYMax = znear * tanf( fovy * JSPI / 360.0f );
    LYMin = -LYMax;

    LXMin = LYMin * aspect;
    LXMax = LYMax * aspect;

    glFrustumf( LXMin, LXMax, LYMin, LYMax, znear, zfar );
}

/** @} Utility */

static void normalize( float *v )
{
    float l2 = v[0] * v[0] + v[1] * v[1] + v[2] * v[2];
    if ( l2 > 0.f )
    {
        float il = 1.f / sqrtf( l2 );
        v[0] *= il;
        v[1] *= il;
        v[2] *= il;
    }
}

/** @addtogroup Utility
 *
 * @{
 */

/**
 * @short Define a viewing transformation.
 *
gluLookAtf() creates a viewing matrix derived from an eye
point, a reference point indicating the center of the scene,
and an UP vector.

The matrix maps the reference point to the negative z axis
and the eye point to the origin. When a typical projection
matrix is used, the center of the scene therefore maps to
the center of the viewport.  Similarly, the direction
described by the UP vector projected onto the viewing plane
is mapped to the positive y axis so that it points upward in
the viewport. The UP vector must not be parallel to the
line of sight from the eye point to the reference point.

Let F be a 3x1 column vector, such that:
@code
      | Xtarget - Xeye |
F = | Ytarget - Yeye |
      | Ztarget - Zeye |
@endcode

Let UP be the up vector <code>(<i>Xup,Yup,Zup</i>)</code>.

Then normalize as follows:
@code
f = F / ||F||
UP' = UP / ||UP||
@endcode

Finally, let s = f x UP', and u = s x f.

M is a 4x4 matrix constructed as follows:
@code
      |  s[0]   s[1]   s[2]  0  |
M = |  u[0]   u[1]   u[2]  0  |
      | -f[0]  -f[1]  -f[2]  0  |
      |    0     0      0    1  |
@endcode

and gluLookAtf() is equivalent to
@code
glMultMatrixf(M);
glTranslatef (-eyex, -eyey, -eyez);
@endcode
 *
 * @param Xeye,Yeye,Zeye Specifies the position of the eye point.
 * @param Xtarget,Ytarget,Ztarget Specifies the position of the reference point.
 * @param Xup,Yup,Zup Specifies the direction of the up vector.
 *
 * @sa gluPerspectivef, glFrustumf
 */
void gluLookAtf( GLfloat Xeye, GLfloat Yeye, GLfloat Zeye,
                 GLfloat Xtarget, GLfloat Ytarget, GLfloat Ztarget,
                 GLfloat Xup, GLfloat Yup, GLfloat Zup )
{
    // Map E-T to z, normalize, x= up^z, normalize, y=z^x, (normalize)
    float z[3] = {Xeye - Xtarget, Yeye - Ytarget, Zeye - Ztarget};
    normalize( z );
    float x[3] = {Yup*z[2] - Zup*z[1], Zup*z[0] - Xup*z[2], Xup*z[1] - Yup*z[0]};
    normalize( x );
    float y[3] = {z[1]*x[2] - z[2]*x[1], z[2]*x[0] - z[0]*x[2], z[0]*x[1] - z[1]*x[0]};
    normalize( y );
    float m[16] =
        {
            x[0], y[0], z[0], 0.f,
            x[1], y[1], z[1], 0.f,
            x[2], y[2], z[2], 0.f,
            0.f, 0.f, 0.f, 1.f,
        };
    glMultMatrixf( m );
    // translate E to 0
    glTranslatef( -Xeye, -Yeye, -Zeye );
}

/**
 * @short Define a 2D orthographic projection matrix.
 *
gluOrtho2Df() sets up a two-dimensional orthographic viewing
region. This is equivalent to calling glOrthof() with near = -1
and far = 1.
 *
 * @param left,right Specify the coordinates for the left and right vertical clipping planes.
 * @param bottom,top Specify the coordinates for the bottom and top horizontal clipping planes.
 *
 * @sa glOrthof, gluPerspectivef
 */
void gluOrtho2Df( GLfloat left, GLfloat right, GLfloat bottom, GLfloat top )
{
    glOrthof( left, right, bottom, top, -1.f, 1.f );
}

/**
 * @short Produce an error string from a GL or GLU error code.
 *
 * gluErrorString() produces an error string from a GL or GLU error code. The string is in ISO Latin 1 format.
 * For example, <code>gluErrorString(GL_OUT_OF_MEMORY)</code> returns the string out of memory.
 *
 * The standard GLU error codes are <code>GLU_INVALID_ENUM</code>, <code>GLU_INVALID_VALUE</code>, 
 * and <code>GLU_OUT_OF_MEMORY</code>. Certain other GLU functions can return specialized error codes through callbacks. See the glGetError() reference page for the list of GL error codes.
 *
 * @param error Specifies a GL or GLU error code. Use glGetError() to retrieve the value of the error flag.
 *
 * @return Returns a human-readable string indicating the error.
 *
 */
const GLubyte* gluErrorString( GLenum error )
{
    switch ( error )
    {
        case GLU_INVALID_ENUM:
        case GL_INVALID_ENUM:
            return ( const GLubyte* )"invalid enum";
        case GLU_INVALID_VALUE:
        case GL_INVALID_VALUE:
            return ( const GLubyte* )"invalid value";
        case GLU_OUT_OF_MEMORY:
        case GL_OUT_OF_MEMORY:
            return ( const GLubyte* )"out of memory";
        case GLU_INVALID_OPERATION:
        case GL_INVALID_OPERATION:
            return ( const GLubyte* )"invalid operation";
        case GL_STACK_OVERFLOW:
            return ( const GLubyte* )"stack overflow";
        case GL_STACK_UNDERFLOW:
            return ( const GLubyte* )"stack underflow";
        default:
            return( NULL );
    }
}

/**
  @short Return a string describing the GLU version or GLU extensions.

gluGetString() returns a pointer to a static string describing
the GLU version or the GLU extensions that are supported.

The version number is one of the following forms:
major_number.minor_number or major_number.minor_number.release_number

The version string is of the following form:
version number<space>vendor-specific information

Vendor-specific information is optional. Its format and
contents depend on the implementation.

The standard GLU contains a basic set of features and
support other features, these may be included as extensions
to the GLU. If name is <CODE>GLU_EXTENSIONS</CODE>, then gluGetString()
returns a space-separated list of names of supported GLU
extensions.  (Extension names never contain spaces.)

All strings are null-terminated.

  @param name Specifies a symbolic constant, one of <CODE>GLU_VERSION</CODE> or <CODE>GLU_EXTENSIONS</CODE>.

  @return Returns a string describing the GLU version or extensions.

  @note
gluGetString() only returns information about GLU extensions.
Call glGetString() to get a list of GL extensions.

  @par Errors
<CODE>NULL</CODE> is returned if name is not <CODE>GLU_VERSION</CODE>
or <CODE>GLU_EXTENSIONS</CODE>.

  @sa glGetString
 */

const GLubyte* gluGetString( GLenum name )
{
    switch ( name )
    {
        case GLU_VERSION:
            return ( const GLubyte* )jsVersion;
        case GLU_EXTENSIONS:
        default:
            return( NULL );
    }
}

/** @} Utility */


//--------------------------------------------------------
// Image manipulation
//--------------------------------------------------------

typedef struct
{
    unsigned char	R, G, B, A;
}
JSRGBA8Pixel;


typedef struct
{
    int	N, TotW, HalfTotW;
    short*	Dat;
    short*	W;
}
JSScaleFilter;


typedef struct
{
    float	Rad, Min, Max;
    float*	Table;
}
JSFilterShape;


static JSFilterShape*	_ShapeBOX = NULL;
static JSFilterShape*	_ShapeGAUSSIAN = NULL;

#define JSScaleSHIFT 		12
#define JSScaleONE 		(1<<JSScaleSHIFT)
#define JSScaleEPSILON		0.0001
#define JSScaleFILTTABSIZE	250


static float _FFloor( const double LFV )
{
    int	LIV;

    LIV = ( int )LFV;

    return(( float )LIV );
}


//================================
// Filter shape functions
//================================
static float _FilterBox( const double LX )
{
    if ( LX < -0.5 ) return( 0.0 );
    if ( LX < 0.5 ) return( 1.0 );
    return( 0.0 );
}


#define GaussNARROWNESS	1.5

float _FilterGaussian( double LX )
{
    LX = LX * GaussNARROWNESS;
    return( 1.0 / exp( LX*LX ) - 1.0 / exp( 1.5*GaussNARROWNESS*1.5*GaussNARROWNESS ) );
}


static float _FilterIntegrate( JSFilterShape* LShape, double LBMin, double LBMax, double LBlurFactor )
{
    float*	LTable;
    float	LMultiplier;
    float	LF1, LF2;
    int	LI1, LI2;


    LBMin /= LBlurFactor;
    LBMax /= LBlurFactor;
    LTable = LShape->Table;
    LMultiplier = ( JSScaleFILTTABSIZE - 1.0 ) / ( 2.0 * LShape->Rad );

    LF1 = (( LBMin - LShape->Min ) * LMultiplier );
    LI1 = ( int )_FFloor( LF1 );
    LF1 = LF1 - LI1;
    if ( LI1 < 0 ) LF1 = 0.0;
    else if ( LI1 >= ( JSScaleFILTTABSIZE - 1 ) ) LF1 = 1.0;
    else LF1 = LTable[LI1] * ( 1.0 - LF1 ) + LTable[LI1+1] * LF1;


    LF2 = (( LBMax - LShape->Min ) * LMultiplier );
    LI2 = ( int )_FFloor( LF2 );
    LF2 = LF2 - LI2;
    if ( LI2 < 0 ) LF2 = 0.0;
    else if ( LI2 >= ( JSScaleFILTTABSIZE - 1 ) ) LF2 = 1.0;
    else LF2 = LTable[LI2] * ( 1.0 - LF2 ) + LTable[LI2+1] * LF2;

    return( LF2 -LF1 );
}


static JSFilterShape* _Integrate( float( *filtfunc )( double ), double LRadius )
{
    JSFilterShape*	LFilter;
    float		LDel, LX, LMin, LMax;
    double		LTotal;
    int		LC;


    LMin = -LRadius;
    LMax = LRadius;
    LDel = LRadius * 2.0;
    LTotal = 0.0;
    LFilter = ( JSFilterShape* )jsMalloc( sizeof( JSFilterShape ) );
    LFilter->Rad = LRadius;
    LFilter->Min = LMin;
    LFilter->Max = LMax;
    LFilter->Table = ( float* )jsMalloc( JSScaleFILTTABSIZE * sizeof( float ) );
    for ( LC = 0;LC < JSScaleFILTTABSIZE;LC++ )
    {
        LX = LMin + ( LDel * LC / ( JSScaleFILTTABSIZE - 1.0 ) );
        LTotal = LTotal + filtfunc( LX );
        LFilter->Table[LC] = LTotal;
    }

    LTotal = 1.0 / LTotal;
    for ( LC = 0;LC < JSScaleFILTTABSIZE;LC++ ) LFilter->Table[LC] *= LTotal;
    return( LFilter );
}


static JSScaleFilter* _MakeFilter( JSFilterShape* LShape, short* LABuf, int LOldSize, int LNewSize, int* LMaxN, const double LBlurFactor )
{
    JSScaleFilter*	LFilter;
    JSScaleFilter*	LFilters;
    float		LBMin, LBMax, LBCent, LBRad;
    float		LFMin, LFMax, LAcent, LARad;
    int		LAMin, LAMax;
    int		LX, LN;
    float		LCoverscale;


    LFilter = LFilters = ( JSScaleFilter* )jsMalloc( LNewSize * sizeof( JSScaleFilter ) );
    *LMaxN = 0;
    if ( LNewSize < LOldSize )
    {
        LCoverscale = (( float )LOldSize / LNewSize * JSScaleONE ) / 2.0;
        LBRad = LBlurFactor * LShape->Rad / LNewSize;
        for ( LX = 0;LX < LNewSize;LX++ )
        {
            LBCent = (( float )LX + 0.5 ) / LNewSize;
            LAMin = ( int )_FFloor(( LBCent - LBRad ) * LOldSize + JSScaleEPSILON );
            LAMax = ( int )_FFloor(( LBCent + LBRad ) * LOldSize - JSScaleEPSILON );
            if ( LAMin < 0 ) LAMin = 0;
            if ( LAMax >= LOldSize ) LAMax = LOldSize - 1;
            LFilter->N = 1 + LAMax - LAMin;
            LFilter->Dat = LABuf + LAMin;
            if ( LFilter->N ) LFilter->W = ( short* )jsMalloc( LFilter->N * sizeof( short ) );
            else LFilter->W = NULL;
            LFilter->TotW = 0;
            for ( LN = 0;LN < LFilter->N;LN++ )
            {
                LBMin = LNewSize * (((( float )LAMin + LN ) / LOldSize ) - LBCent );
                LBMax = LNewSize * (((( float )LAMin + LN + 1 ) / LOldSize ) - LBCent );
                LFilter->W[LN] = ( short )_FFloor(( LCoverscale * _FilterIntegrate( LShape, LBMin, LBMax, LBlurFactor ) ) + 0.5 );
                LFilter->TotW += LFilter->W[LN];
            }
            if ( LFilter->TotW < 1 ) LFilter->TotW = 1;
            LFilter->HalfTotW = LFilter->TotW / 2;
            if ( LFilter->N > *LMaxN ) *LMaxN = LFilter->N;
            LFilter++;
        }
    }
    else
    {
        LCoverscale = (( float )LNewSize / LOldSize * JSScaleONE ) / 2.0;
        LARad = LBlurFactor * LShape->Rad / LOldSize;
        for ( LX = 0;LX < LNewSize;LX++ )
        {
            LBMin = (( float )LX ) / LNewSize;
            LBMax = (( float )LX + 1.0 ) / LNewSize;
            LAMin = ( int )_FFloor(( LBMin - LARad ) * LOldSize + ( 0.5 + JSScaleEPSILON ) );
            LAMax = ( int )_FFloor(( LBMax + LARad ) * LOldSize - ( 0.5 + JSScaleEPSILON ) );
            if ( LAMin < 0 ) LAMin = 0;
            if ( LAMax >= LOldSize ) LAMax = LOldSize - 1;
            LFilter->N = 1 + LAMax - LAMin;
            LFilter->Dat = LABuf + LAMin;

            if ( LFilter->N ) LFilter->W = ( short* )jsMalloc( LFilter->N * sizeof( short ) );
            else LFilter->W = NULL;

            LFilter->TotW = 0;
            for ( LN = 0;LN < LFilter->N;LN++ )
            {
                LAcent = ( LAMin + LN + 0.5 ) / LOldSize;
                LFMin = LOldSize * ( LBMin - LAcent );
                LFMax = LOldSize * ( LBMax - LAcent );
                LFilter->W[LN] = ( short )_FFloor(( LCoverscale * _FilterIntegrate( LShape, LFMin, LFMax, LBlurFactor ) ) + 0.5 );
                LFilter->TotW += LFilter->W[LN];
            }
            if ( LFilter->TotW < 1 ) LFilter->TotW = 1;
            LFilter->HalfTotW = LFilter->TotW / 2;
            if ( LFilter->N > *LMaxN )*LMaxN = LFilter->N;
            LFilter++;
        }
    }
    return LFilters;
}


//========================================
// Free a Filter
//========================================
static void _FilterFree( JSScaleFilter* LFilt, int LN )
{
    JSScaleFilter*	LF;

    LF = LFilt;
    while ( LN-- ) { if ( LF->W ) jsFree( LF->W );LF++; }
    jsFree( LFilt );
}



static void _RowSet( int* LBuffer, int LVal, int LN )
{
    while ( LN-- ) *LBuffer++ = LVal;
}


static void _RowAdd( int* LDPtr, const short* LSPtr, int LW, int LN )
{
    while ( LN-- )( *LDPtr++ ) += ( LW * ( *LSPtr++ ) );
}


static void _RowDivide( int* LSPtr, short* LDPtr, int LTotal, int LN )
{
    while ( LN-- ) *( LDPtr++ ) = *( LSPtr++ ) / LTotal;
}


static void _ApplyXFilter( short* LBBuf, const JSScaleFilter* LXFilter, int LBNX )
{
    short*	LW;
    short*	LDPtr;
    int	LN, LVal;


    if (( LN = LXFilter->N ) == 1 ) { while ( LBNX-- ) { *LBBuf++ = *LXFilter->Dat;LXFilter++; } }
    else
    {
        while ( LBNX-- )
        {
            LW = LXFilter->W;
            LDPtr = LXFilter->Dat;
            LVal = LXFilter->HalfTotW;
            LN = LXFilter->N;
            while ( LN-- ) LVal += ( *LW++ ) * ( *LDPtr++ );
            *LBBuf++ = LVal / LXFilter->TotW;
            LXFilter++;
        }
    }
}


static int _DoScalingUBYTE( int LNumOfChannels, GLsizei LXSize, GLsizei LYSize, const void* dataIn, GLsizei LNewXSize, GLsizei LNewYSize, JSRGBA8Pixel* dataOut )
{
    JSFilterShape*	LFilterShape = _ShapeGAUSSIAN;
// JSFilterShape*	LFilterShape = _ShapeBOX;
    JSScaleFilter*	LFilter;
    JSScaleFilter*	LXFilt;
    JSScaleFilter*	LYFilt;
    int*		LAccRow = NULL;
    short*		LLineBuffer = ( short* )jsMalloc( LNewXSize * sizeof( short ) );
    short*		LAXBuf;
    short*		LBXBuf;
    short*		LTBuf;
    short*		LRow;
    short**	LFilterRows;
    double		LBlurFactor = 0.5;
    int		LNumOfRows;

    short*		LLineBufferD;
    int		LX, LY;
    int		LCh,
    LCurAY, LZY = 0, LZAY = 0,
                            LMax;
    unsigned int	LC, LN,
    LLineSize = LNewXSize * sizeof( short ),
                LDLineOffset;


    if ( LLineBuffer == NULL ) return( GLU_OUT_OF_MEMORY );

// For scaling up, use a larger window
//
    if (( LNewXSize > LXSize ) || ( LNewYSize > LYSize ) ) LBlurFactor = 1.6;else LBlurFactor = 0.5;

    LAXBuf = ( short* )jsMalloc( LXSize * sizeof( short ) );
    LBXBuf = ( short* )jsMalloc( LNewXSize * sizeof( short ) );
    LTBuf = ( short* )jsMalloc( LNewXSize * sizeof( short ) );
    LXFilt = _MakeFilter( LFilterShape, LAXBuf, LXSize, LNewXSize, &LNumOfRows, LBlurFactor );
    LYFilt = _MakeFilter( LFilterShape, 0, LYSize, LNewYSize, &LNumOfRows, LBlurFactor );
    LFilterRows = ( short** )jsMalloc( LNumOfRows * sizeof( short* ) );
    for ( LX = 0;LX < LNumOfRows;LX++ ) LFilterRows[LX] = ( short* )jsMalloc( LNewXSize * sizeof( short ) );
    LAccRow = ( int* )jsMalloc( LNewXSize * sizeof( int ) );


// RGBA8
//
    {
        JSRGBA8Pixel*		LSrcImage = ( JSRGBA8Pixel* )dataIn;
        JSRGBA8Pixel*		LSrcPixel;
        JSRGBA8Pixel*		LDstImage = ( JSRGBA8Pixel* )dataOut;
        JSRGBA8Pixel*		LDstPixel;


        for ( LCh = 0;LCh < LNumOfChannels;LCh++ )
        {
            for ( LY = 0, LDLineOffset = 0;LY < LNewYSize;LY++, LDLineOffset += LNewXSize )
            {
//    if(!(LScaleRec->GoScale)) break;
                if ( LY == 0 ) { LCurAY = -1;LZY = 0;LZAY = 0; }

                {
                    if (( LYSize == 1 ) && ( LNewYSize == 1 ) )	// Just a line
                    {
                        LSrcPixel = LSrcImage + LZAY * LXSize;LLineBufferD = LAXBuf;LZAY++;
                        switch ( LCh )
                        {
                            case 0:	for ( LC = 0;LC < ( unsigned int )LXSize;LC++, LSrcPixel++ ) LLineBufferD[LC] = LSrcPixel->R;break;
                            case 1:	for ( LC = 0;LC < ( unsigned int )LXSize;LC++, LSrcPixel++ ) LLineBufferD[LC] = LSrcPixel->G;break;
                            case 2:	for ( LC = 0;LC < ( unsigned int )LXSize;LC++, LSrcPixel++ ) LLineBufferD[LC] = LSrcPixel->B;break;
                            case 3:	for ( LC = 0;LC < ( unsigned int )LXSize;LC++, LSrcPixel++ ) LLineBufferD[LC] = LSrcPixel->A;break;
                        }

                        _ApplyXFilter( LFilterRows[0], LXFilt, LNewXSize );
                        memcpy( LLineBuffer, LFilterRows[0], LLineSize );
                    }
                    else
                    {
                        LFilter = LYFilt + LZY;
                        LMax = ( LFilter->Dat - ( short * )NULL ) + ( LFilter->N - 1 );
                        while ( LZAY <= LMax )
                        {
                            LSrcPixel = LSrcImage + LZAY * LXSize;LLineBufferD = LAXBuf;LZAY++;
                            switch ( LCh )
                            {
                                case 0:	for ( LC = 0;LC < ( unsigned int )LXSize;LC++, LSrcPixel++ ) LLineBufferD[LC] = LSrcPixel->R;break;
                                case 1:	for ( LC = 0;LC < ( unsigned int )LXSize;LC++, LSrcPixel++ ) LLineBufferD[LC] = LSrcPixel->G;break;
                                case 2:	for ( LC = 0;LC < ( unsigned int )LXSize;LC++, LSrcPixel++ ) LLineBufferD[LC] = LSrcPixel->B;break;
                                case 3:	for ( LC = 0;LC < ( unsigned int )LXSize;LC++, LSrcPixel++ ) LLineBufferD[LC] = LSrcPixel->A;break;
                            }
                            LRow = LFilterRows[0];
                            for ( LC = 0;LC < ( unsigned int )( LNumOfRows - 1 );LC++ ) LFilterRows[LC] = LFilterRows[LC+1];
                            LFilterRows[LNumOfRows-1] = LRow;
                            _ApplyXFilter( LFilterRows[LNumOfRows-1], LXFilt, LNewXSize );
                        }
                        if ( LFilter->N == 1 )
                        {
                            memcpy( LLineBuffer, LFilterRows[LNumOfRows-1], LLineSize );
                        }
                        else
                        {
                            _RowSet( LAccRow, LFilter->HalfTotW, LNewXSize );
                            for ( LC = 0, LN = LFilter->N;LC < LN;LC++ ) _RowAdd( LAccRow, LFilterRows[LC+( LNumOfRows-1 )-( LFilter->N-1 )], LFilter->W[LC], LNewXSize );
                            _RowDivide( LAccRow, LBXBuf, LFilter->TotW, LNewXSize );
                            memcpy( LLineBuffer, LBXBuf, LLineSize );
                        }
                    }
                }
                LZY++;


                LDstPixel = LDstImage + LDLineOffset;
                switch ( LCh )
                {
                    case 0:
                        for ( LX = 0;LX < LNewXSize;LX++, LDstPixel++ ) LDstPixel->R = ( unsigned char )( LLineBuffer[LX] );
                        break;

                    case 1:
                        for ( LX = 0;LX < LNewXSize;LX++, LDstPixel++ ) LDstPixel->G = ( unsigned char )( LLineBuffer[LX] );
                        break;

                    case 2:
                        for ( LX = 0;LX < LNewXSize;LX++, LDstPixel++ ) LDstPixel->B = ( unsigned char )( LLineBuffer[LX] );
                        break;

                    case 3:
                        for ( LX = 0;LX < LNewXSize;LX++, LDstPixel++ ) LDstPixel->A = ( unsigned char )( LLineBuffer[LX] );
                        break;
                }
            }
        }
    }


    if ( LAccRow ) jsFree( LAccRow );
    if ( LLineBuffer ) jsFree( LLineBuffer );
    if ( LAXBuf ) jsFree( LAXBuf );
    if ( LBXBuf ) jsFree( LBXBuf );
    if ( LTBuf ) jsFree( LTBuf );
    if ( LXFilt ) _FilterFree( LXFilt, LNewXSize );
    if ( LYFilt ) _FilterFree( LYFilt, LNewYSize );
    if ( LFilterRows )
    {
        for ( LX = 0;LX < LNumOfRows;LX++ ) jsFree( LFilterRows[LX] );
        jsFree( LFilterRows );
    }

    return( 0 );
}

/** @addtogroup Utility
 *
 * @{
 */

/**
  @short Scale an image to an arbitrary size with a box filter

gluScaleImage() scales a pixel image using the appropriate
pixel store modes to unpack data from the source image and
pack data into the destination image.

When shrinking an image, gluScaleImage() uses a box filter to
sample the source image and create pixels for the
destination image. When magnifying an image, the pixels from
the source image are linearly interpolated to create the
destination image.

  @param format Specifies the format of the pixel data.
                The following symbolic values are valid:
		<CODE>GL_RGBA</CODE>, <CODE>GL_BGRA</CODE>, and <CODE>GL_ARGB_SCE</CODE>.
  @param  LXSize,LYSize  Specify the width and height, respectively, of the source image	that is	scaled.
  @param  typeIn  Specifies the data type for <I><c>dataIn</c></I>. Must be <CODE>GL_UNSIGNED_BYTE</CODE>.
  @param  dataIn  Specifies a pointer to the source image.
  @param  LNewXSize,LNewYSize  Specify the width and height, respectively, of the destination image.
  @param  typeOut Specifies the data type for <I><c>dataOut</c></I>. Must be <CODE>GL_UNSIGNED_BYTE</CODE>.
  @param  dataOut Specifies a pointer to the destination image.

  @return Returns a value of 0 indicating success; otherwise a GLU error
code is returned (see gluErrorString()).

  @par Errors
Use glGetError() to retrieve the value of the error flag.
<TABLE>
<TR>
<TD><CODE>GLU_INVALID_VALUE</CODE></TD>
<TD><I><c>LXSize</c></I>, <I><c>LYSize</c></I>, <I><c>LNewXSize</c></I>, or <I><c>LNewYSize</c></I> are < 0.</TD>
</TR>
<TR>
<TD><CODE>GLU_INVALID_ENUM</CODE></TD>
<TD><I><c>format</c></I>, <I><c>typeIn</c></I>, or <I><c>typeOut</c></I> are not legal.</TD>
</TR>
</TABLE>

  @sa gluBuild2DMipmaps, gluErrorString
 */

GLint gluScaleImage( GLenum format, GLsizei LXSize, GLsizei LYSize, GLenum typeIn, const void* dataIn, GLsizei LNewXSize, GLsizei LNewYSize, GLenum typeOut, GLvoid* dataOut )
{
    int	LNumOfChannels = 0;


    if (( LXSize < 1 ) || ( LYSize < 1 ) ) return( GLU_INVALID_VALUE );


    if ( !_ShapeBOX ) _ShapeBOX = _Integrate( _FilterBox, 0.5 );
    if ( !_ShapeGAUSSIAN ) _ShapeGAUSSIAN = _Integrate( _FilterGaussian, 1.5 );


    switch ( format )
    {
        case GL_RGBA:
        case GL_BGRA:
        case GL_ARGB_SCE:
            LNumOfChannels = 4;
            break;
        default:
            return( GLU_INVALID_ENUM );
    }

    JS_ASSERT( LNumOfChannels );

    switch ( typeIn )
    {
        case GL_UNSIGNED_BYTE:
            break;
        default:
            return( GLU_INVALID_ENUM );
    }

    switch ( typeOut )
    {
        case GL_UNSIGNED_BYTE:
            break;
        default:
            return( GLU_INVALID_ENUM );
    }

    return( _DoScalingUBYTE( LNumOfChannels, LXSize, LYSize, ( JSRGBA8Pixel* )dataIn, LNewXSize, LNewYSize, ( JSRGBA8Pixel* )dataOut ) );
}

/**
 *  @}  Utility
 */

int	_2DTextureSizes[] = { 16, 32, 64, 128, 256, 512, 1024, 2048, 4096, 0 }
                        ;		// List must end with a 0

/** @addtogroup Utility
 *
 *  @{
 */
/**
  @short Builds a 2-D mipmap

gluBuild2DMipmaps() builds a series of prefiltered 2-D texture
	  maps of decreasing resolutions called	a mipmap. This is used
	  for the antialiasing of texture mapped primitives.

	  Initially, the width and height of data are checked to see
	  if they are a	power of two. If not, a	copy of	data (not
	  data), is scaled up or down to the nearest power of two.
	  This copy will be used for subsequent	mipmapping operations
	  described below. (If width or	height is exactly between
	  powers of 2, then the	copy of	data will scale	upwards.)  For
	  example, if width is 57 and height is	23 then	a copy of data
	  will scale up	to 64 and down to 16, respectively, before
	  mipmapping takes place.

	  Then,	proxy textures (see glTexImage2D()) are used to
	  determine if the implementation can fit the requested
	  texture. If not, both	dimensions are continually halved
	  until	it fits. (If the OpenGL	version	is <= 1.0, both
	  maximum texture dimensions are clamped to the	value returned
	  by glGetIntegerv() with the argument <CODE>GL_MAX_TEXTURE_SIZE</CODE>.)

	  Next,	a series of mipmap levels is built by decimating a
	  copy of data in half along both dimensions until size	1x1 is
	  reached. At each level, each texel in	the halved mipmap
	  level	is an average of the corresponding four	texels in the
	  larger mipmap	level. (In the case of rectangular images, the
	  decimation will ultimately reach an N	x 1 or 1 x N
	  configuration. Here, two texels are averaged instead.)

	  glTexImage2D() is called to load each of these mipmap levels.
	  Level	0 is a copy of data. The highest level is
	  log2(max(width,height)).  For	example, if width is 64	and
	  height is 16 and the implementation can store	a texture of
	  this size, the following mipmap levels are built: 64x16,
	  32x8,	16x4, 8x2, 4x1,	2x1 and	1x1. These correspond to
	  levels 0 through 6, respectively.

  @param target   Specifies the	target texture.	Must be  <CODE>GL_TEXTURE_2D</CODE>.
  @param internalFormat  Specifies the color components in the texture.
The following symbolic values are accepted: <CODE>GL_RGBA</CODE>,
<CODE>GL_BGRA</CODE>, <CODE>GL_LUMINANCE</CODE>, and <CODE>GL_LUMINANCE_ALPHA</CODE>.

  @param  LXSize,LYSize  Specify the width and height, respectively, of the source image that is scaled.
  @param  format  Specifies the	format of the pixel data. Must be one of:  <CODE>GL_RGBA</CODE>,
<CODE>GL_BGRA</CODE>, <CODE>GL_LUMINANCE</CODE>, and <CODE>GL_LUMINANCE_ALPHA</CODE>.
  @param  type  Specifies the data type for <I><c>data</c></I>. Must be <CODE>GL_UNSIGNED_BYTE</CODE>.
  @param  data  Specifies a pointer to the image data in memory.

  @return Returns a value of 0 indicating success; otherwise a GLU error
code is returned (see gluErrorString()).

  @par Errors
Use glGetError() to retrieve the value of the error flag.
<TABLE>
<TR>
<TD><CODE>GLU_INVALID_VALUE</CODE></TD>
<TD><I><c>LXSize</c></I> or <I><c>LYSize</c></I> are < 1.</TD>
</TR>
<TR>
<TD><CODE>GLU_INVALID_ENUM</CODE></TD>
<TD><I><c>internalFormat</c></I>, <I><c>format</c></I>, or <I><c>type</c></I> are not legal.</TD>
</TR>
</TABLE>

  @sa glTexImage2D, gluErrorString, gluScaleImage
 */


GLint gluBuild2DMipmaps( GLenum target, GLint internalFormat,
                         GLsizei LXSize, GLsizei LYSize,
                         GLenum format, GLenum type, const void* data )
{
    if (( target != GL_TEXTURE_2D ) || ( LXSize < 1 ) || ( LYSize < 1 ) )
        return( GLU_INVALID_VALUE );

    {
        void*		LCorrectedStartImage = NULL;
        int		LNewXSize = 0, LNewYSize = 0,
                                        LD, LDiff,
                                        LLevels;
        unsigned int	LC, LNumOfChannels = 0, LComponentSize = 0, LPixelSize = 0;


        switch ( format )
        {
            case GL_RED:
            case GL_GREEN:
            case GL_BLUE:
            case GL_ALPHA:
                break;

            case GL_RGB:			break;

            case GL_RGBA:		LNumOfChannels = 4;break;
            case GL_BGR:			break;
            case GL_BGRA:		LNumOfChannels = 4;break;
            case GL_LUMINANCE:		LNumOfChannels = 1;break;
            case GL_LUMINANCE_ALPHA:	LNumOfChannels = 2;break;

            default:	return( GLU_INVALID_ENUM );
        }

        if ( LNumOfChannels )
        {
            switch ( type )
            {
                case GL_UNSIGNED_BYTE:
                    LComponentSize = 1;
                    break;

                case GL_BYTE:
                case GL_UNSIGNED_SHORT:
                case GL_SHORT:
                case GL_UNSIGNED_INT:
                case GL_INT:
                case GL_FLOAT:
                case GL_UNSIGNED_BYTE_3_3_2:
                case GL_UNSIGNED_BYTE_2_3_3_REV:
                case GL_UNSIGNED_SHORT_5_6_5:
                case GL_UNSIGNED_SHORT_5_6_5_REV:
                case GL_UNSIGNED_SHORT_4_4_4_4:
                case GL_UNSIGNED_SHORT_4_4_4_4_REV:
                case GL_UNSIGNED_SHORT_5_5_5_1:
                case GL_UNSIGNED_SHORT_1_5_5_5_REV:
                case GL_UNSIGNED_INT_8_8_8_8:
                case GL_UNSIGNED_INT_8_8_8_8_REV:
                case GL_UNSIGNED_INT_10_10_10_2:
                case GL_UNSIGNED_INT_2_10_10_10_REV:
                    break;

                default:	return( GLU_INVALID_ENUM );
            }
        }

        LPixelSize = LNumOfChannels * LComponentSize;

        if ( LPixelSize == 0 ) return( GLU_INVALID_VALUE );

        // Make sure the dimensions of the input image are powers of 2
        //
    for ( LC = 0;_2DTextureSizes[LC] != 0;LC++ ) if ( LXSize == _2DTextureSizes[LC] ) { LNewXSize = LXSize;break; }
        for ( LC = 0;_2DTextureSizes[LC] != 0;LC++ ) if ( LYSize == _2DTextureSizes[LC] ) { LNewYSize = LYSize;break; }

        if ( LNewXSize == 0 )
        {
            if ( LXSize < _2DTextureSizes[0] ) LNewXSize = _2DTextureSizes[0];
            else
            {
                LDiff = LXSize - _2DTextureSizes[0];
                for ( LC = 1;_2DTextureSizes[LC] != 0;LC++ )
                {
                    LD = LXSize - _2DTextureSizes[LC];
                    if ( abs( LD ) < abs( LDiff ) ) LDiff = LD;
                }
                LNewXSize = LXSize - LDiff;
            }
        }

        if ( LNewYSize == 0 )
        {
            if ( LYSize < _2DTextureSizes[0] ) LNewYSize = _2DTextureSizes[0];
            else
            {
                LDiff = LYSize - _2DTextureSizes[0];
                for ( LC = 1;_2DTextureSizes[LC] != 0;LC++ )
                {
                    LD = LYSize - _2DTextureSizes[LC];
                    if ( abs( LD ) < abs( LDiff ) ) LDiff = LD;
                }
                LNewYSize = LYSize - LDiff;
            }
        }



        // Now check if we have to scale the input image
        //
        if (( LNewXSize != LXSize ) || ( LNewYSize != LYSize ) )
        {
            LCorrectedStartImage = jsMalloc( LPixelSize * LNewXSize * LNewYSize );

            gluScaleImage( format, LXSize, LYSize, type, data, LNewXSize, LNewYSize, type, LCorrectedStartImage );

            LXSize = LNewXSize;
            LYSize = LNewYSize;
            glTexImage2D( target, 0, internalFormat, LXSize, LYSize, 0, format, type, LCorrectedStartImage );
        }
        else
        {
#ifdef GLU_DEBUG
            printf( "%s() LEVEL 0 %dx%d\n", __func__, LXSize, LYSize );fflush( stdout );
#endif
            glTexImage2D( target, 0, internalFormat, LXSize, LYSize, 0, format, type, data );
        }



        // Calculate max. possible mipmap levels
        //
        for ( LLevels = 0;( LXSize > 1 ) || ( LYSize > 1 );LLevels++ )
        {
            if ( LXSize > 1 ) LXSize >>= 1;
            if ( LYSize > 1 ) LYSize >>= 1;
        }
        if ( LLevels < 1 ) return( 0 );

#ifdef GLU_DEBUG
        printf( "%s() LNumOfChannels %d  Max levels from %dx%d : %d\n", __func__, LNumOfChannels, LXSize, LYSize, LLevels );fflush( stdout );
#endif

        {
            void*	LTmpImageBuffer;
            void*	LImageBuffer2;
            void*	LOldImageBuffer;
            void*	LNewImageBuffer;
            unsigned int	LBytesPerPixel = LComponentSize * LNumOfChannels,
                                          LOldXSize, LOldYSize;


            LXSize = LNewXSize;
            LYSize = LNewYSize;

#ifdef GLU_DEBUG
            printf( "%s() LEVEL 0 %dx%d %d\n", __func__, LXSize, LYSize, LBytesPerPixel );fflush( stdout );
#endif
            LImageBuffer2 = jsMalloc((( LXSize * LYSize ) >> 2 ) * LBytesPerPixel );
            LNewImageBuffer = LImageBuffer2;


            // We will only need 2 image buffers for the successive down-scaling and
            // if we had to scale the input to the closest powers of 2, we already
            // have one buffer.
            //
            if ( LCorrectedStartImage )
            {
                LC = 0;

                LOldImageBuffer = LCorrectedStartImage;
            }
            else
            {
                // If we didn't have to scale the original image, we won't have an extra buffer to play with...
                //
                LC = 1;


                LOldXSize = LXSize;
                LOldYSize = LYSize;

                if ( LXSize > 1 ) LXSize >>= 1;
                if ( LYSize > 1 ) LYSize >>= 1;

                gluScaleImage( format, LOldXSize, LOldYSize, type, data, LXSize, LYSize, type, LNewImageBuffer );

                glTexImage2D( GL_TEXTURE_2D, 1, internalFormat, LXSize, LYSize, 0, format, type, LNewImageBuffer );

#ifdef GLU_DEBUG
                printf( "%s() LEVEL 1 %dx%d\n", __func__, LXSize, LYSize );fflush( stdout );
#endif

                LOldImageBuffer = LNewImageBuffer;

                LNewImageBuffer = LCorrectedStartImage = jsMalloc((( LXSize * LYSize ) >> 2 ) * LBytesPerPixel );
            }


            for ( ;LC < ( unsigned int )LLevels;LC++ )
            {
                LOldXSize = LXSize;
                LOldYSize = LYSize;

                if ( LXSize > 1 ) LXSize >>= 1;
                if ( LYSize > 1 ) LYSize >>= 1;

                gluScaleImage( format, LOldXSize, LOldYSize, type, LOldImageBuffer, LXSize, LYSize, type, LNewImageBuffer );

                glTexImage2D( GL_TEXTURE_2D, LC + 1, internalFormat, LXSize, LYSize, 0, format, type, LNewImageBuffer );

                // Swap source and destination buffers
                //
                LTmpImageBuffer = LOldImageBuffer;
                LOldImageBuffer = LNewImageBuffer;
                LNewImageBuffer = LTmpImageBuffer;

#ifdef GLU_DEBUG
                printf( "%s() Level %d:  %dx%d\n", __func__, LC + 1, LXSize, LYSize );fflush( stdout );
#endif
            }


            if ( LImageBuffer2 ) jsFree( LImageBuffer2 );
        }


        if ( LCorrectedStartImage ) jsFree( LCorrectedStartImage );
    }

    return( 0 );
}


/** @} Utility */
