/* SCE CONFIDENTIAL
 * PlayStation(R)3 Programmer Tool Runtime Library 360.001
 * Copyright (C) 2010 Sony Computer Entertainment Inc.
 * All Rights Reserved.
 */

#ifndef __CELL_FLASH_H__
#define __CELL_FLASH_H__

#include <sys/types.h>
#include <flash/error.h>

/**
 * @file flash.h 
 * @jp  Flash Library API @endjp
 * @en  Flash Library API @enden
 */

#if defined(__cplusplus)
extern "C" {
#endif

	//**********************************************************************
	//* types
	//**********************************************************************
	typedef int CellFlashBool;
	typedef uint32_t CellFlashError;
	typedef void* CellFlashIdent;

	typedef enum CellFlashScriptAccess { 
		CELL_FLASH_SCRIPT_ACCESS_NEVER = 0,
		CELL_FLASH_SCRIPT_ACCESS_ALWAYS, 
		CELL_FLASH_SCRIPT_ACCESS_SAME_DOMAIN ,
		CELL_FLASH_SCRIPT_ACCESS_LIM_ = 0xffffffff // reserved: you don't use this value
	} CellFlashScriptAccess;

	typedef enum CellFlashQuality {
		CELL_FLASH_QUALITY_LOW = 0,
		CELL_FLASH_QUALITY_MEDIUM,
		CELL_FLASH_QUALITY_HIGH,
		CELL_FLASH_QUALITY_LIM_ = 0xffffffff // reserved: you don't use this value
	} CellFlashQuality;

	typedef enum CellFlashScale {
		CELL_FLASH_SCALE_SHOW_ALL = 0,
		CELL_FLASH_SCALE_NO_BORDER = 1,
		CELL_FLASH_SCALE_EXACT_FIT = 2,
		CELL_FLASH_SCALE_NO_SCALE = 3,
		CELL_FLASH_SCALE_LIM_ = 0xffffffff // reserved: you don't use this value
	} CellFlashScale;

	typedef enum CellFlashAlign {
		CELL_FLASH_ALIGN_NO_ALIGN = 0,
		CELL_FLASH_ALIGN_LEFT =0x10,
		CELL_FLASH_ALIGN_RIGHT = 0x20,
		CELL_FLASH_ALIGN_TOP = 0x30,
		CELL_FLASH_ALIGN_BOTTOM = 0x40,
		CELL_FLASH_ALIGN_LIM_ = 0xffffffff // reserved: you don't use this value
	} CellFlashAlign;

	typedef enum CellFlashCursor{
		CELL_FLASH_CURSOR_NONE,
		CELL_FLASH_CURSOR_BLANK,
		CELL_FLASH_CURSOR_ARROW, ///< normal
		CELL_FLASH_CURSOR_BUTTON, ///< clickable
		CELL_FLASH_CURSOR_HAND, 
		CELL_FLASH_CURSOR_I, ///< I-beam
		CELL_FLASH_CURSOR_LIM_ = 0xffffffff // reserved: you don't use this value
	} CellFlashCursor;
	
	typedef struct CellFlashParamset {
		int x; ///< could be zero
		int y; ///< could be zero
		int width; ///< in pixel
		int height; ///< in pixel 
		uint32_t bgcolor; ///< background color in ARGB
		CellFlashBool loop; ///< auto-loopable
		CellFlashBool play; ///< auto-playable
		CellFlashBool allowNetworking; ///< if it uses network
		CellFlashBool allowFullscreen; ///< if it uses fullscreen (unsupported)
		CellFlashScriptAccess allowScriptAccess; ///< if it allows ExternalInterface
		CellFlashQuality quality;
		CellFlashScale scale;
		CellFlashAlign align;
	} CellFlashParamset ;

	typedef struct CellFlashCoreParamset {
		void* cellSpurs; /// CellSpurs
		int spuNums; /// num of spu (1-5). if you use AVC codec, number should be over 3 up to 5.
		int main_thread_prio;   /// base priority for all
		int fio_thread_prio;    /// local file stream
		int stream_thread_prio; /// general stream (message/http/FDAP/crossdomain.xml)
		int socket_thread_prio; /// socket
		int timer_thread_prio; /// timer thread: usually high
		int sound_thread_prio; /// sound thread: the consumer of sound.
		int soundmix_thread_prio; /// sound mixir: the producer of sound.
		int vdec_thread_prio; /// video decoder H264 and VP6
		int adec_thread_prio; /// audio decoder mp3 and aac
		int gcElapsed; /// unused
		void* stack_base;
		int stack_size;
		const char* tempBase; /// the path to your temp dir
		const char* lang; /// language string (up to 32 bytes. "LC", "english", "japanese", ...) for the environment: see flash.system.Capability
	} CellFlashCoreParamset ;

	typedef struct CellFlashInstance {
		void* player;
		void* host;
	} CellFlashInstance;

	/**
	 * @enum CellFlashVar
	 * 
	 * @jp
	 * ホスト変数の識別子
	 * プレイヤーがホストアプリの状態を取得する際に使用されます。
	 * ブラウザを対象とした仕様のもので、通常気にすることはないかも知れません。
	 * @endjp
	 * @en
	 * the ideintifiers for accessing to the host-application's variable.
	 * this is NPAPI specific identifiers. you may no need care this.
	 * @enden
	 */
	typedef enum CellFlashVar {
		CELL_FLASH_VAR_NAME_STRING = 1,
		CELL_FLASH_VAR_DESC_STRING,
		CELL_FLASH_VAR_WINDOW_BOOL,
		CELL_FLASH_VAR_TRANSPARENT_BOOL,
		CELL_FLASH_VAR_TIMER_INTERVAL = 7,
		CELL_FLASH_VAR_SCRIPTABLE_OBJECT  = 15,
		CELL_FLASH_VAR_LIM_ = 0xffffffff // reserved: you don't use this value
	} CellFlashVar;

	/**
	 * @enum CellFlashHostVar
	 * 
	 * @jp
	 * ホスト変数の識別子
	 * プレイヤーがホストアプリの状態を取得する際に使用されます。
	 * ブラウザを対象とした仕様のもので、通常気にすることはないかも知れません。
	 * @endjp
	 * @en
	 * the ideintifiers for accessing to the host-application's variable.
	 * this is NPAPI specific identifiers. you may do not need care this.
	 * @enden
	 */
	typedef enum CellFlashHostVar{
		CELL_FLASH_HOST_VAR_DISPLAY = 1,
		CELL_FLASH_HOST_VAR_APP_CONTEXT,
		CELL_FLASH_HOST_VAR_WINDOW,
		CELL_FLASH_HOST_VAR_JS_ENABLED_BOOL,
		CELL_FLASH_HOST_VAR_ASD_ENABLED_BOOL,
		CELL_FLASH_HOST_VAR_IS_OFFLINE_BOOL,
		CELL_FLASH_HOST_VAR_WINDOW_OBJECT = 15,
		CELL_FLASH_HOST_VAR_ELEMENT_OBJECT = 16,
		CELL_FLASH_HOST_VAR_LIM_ = 0xffffffff // reserved: you don't use this value
	} CellFlashHostVar;

	typedef void* CellFlashStream;

	typedef int CellFlashMimetype;

	typedef struct CellFlashRange {
		int32_t offset;
		uint32_t length;
		struct CellFlashRange_* next;
	} CellFlashRange;

	typedef struct CellFlashRect {
		uint16_t top;
		uint16_t left;
		uint16_t bottom;
		uint16_t right;
	} CellFlashRect;

	typedef void* CellFlashRegion;

	typedef const char* CellFlashUserAgent;

	typedef void* CellFlashAtom;

	typedef struct CellFlashString {
		const char* utf8characters;
		int utf8length;
	} CellFlashString;

	typedef struct CellFlashArray {
		int length;
		void* ptr;
	} CellFlashArray;

	typedef struct CellFlashClass {
		typedef void* (*allocatorFunc)(size_t l, void* opt);
		typedef void (*deallocatorFunc)(void* p);
		typedef void (*invalidateFunc)(void* p);
		typedef CellFlashBool (*hasMethodFunc)(void* p);
		typedef CellFlashBool (*invokeFunc)(void* p);
		typedef CellFlashBool (*defaultInvokeFunc)(void* p);
		typedef CellFlashBool (*hasPropFunc)(void* p);
		typedef CellFlashBool (*getPropFunc)(void* p, void* val);
		typedef CellFlashBool (*setPropFunc)(void* p, void* val);
		typedef CellFlashBool (*delPropFunc)(void* p);

		uint32_t structVersion;

		allocatorFunc allocate;
		deallocatorFunc deallocate;
		invalidateFunc invalidate;
		hasMethodFunc hasMethod;
		invokeFunc invoke;
		defaultInvokeFunc invokeDefault;
		hasPropFunc hasProperty;
		getPropFunc getProperty;
		setPropFunc setProperty;
		delPropFunc removeProperty;
	} CellFlashClass;

	typedef struct CellFlashObject {
		CellFlashClass* class_;
		uint32_t refCount_;
	} CellFlashObject;

	/**
	 * @enum CellFlashSoundSampleType
	 * @jp PCMサンプルの型 @endjp @en the type of PCM sample @enden
	 */
	typedef enum CellFlashSoundSampleType {
		CELL_FLASH_SOUND_SAMPLE_TYPE_USHORT=1,
		CELL_FLASH_SOUND_SAMPLE_TYPE_SSHORT,
		CELL_FLASH_SOUND_SAMPLE_TYPE_UINT,
		CELL_FLASH_SOUND_SAMPLE_TYPE_SINT,
		CELL_FLASH_SOUND_SAMPLE_TYPE_FLOAT,
		CELL_FLASH_SOUND_SAMPLE_TYPE_LIM_ = 0xffffffff // reserved: you don't use this value
	} CellFlashSoundSampleType;
	
	/**
	 * @enum CellFlashVariantType
	 * @jp バリアントの型 @endjp @en a type of variant @enden
	 */
	typedef enum CellFlashVariantType {
		CELL_FLASH_VARIANT_TYPE_VOID,
		CELL_FLASH_VARIANT_TYPE_NULL,
		CELL_FLASH_VARIANT_TYPE_BOOL,
		CELL_FLASH_VARIANT_TYPE_INT32,
		CELL_FLASH_VARIANT_TYPE_DOUBLE,
		CELL_FLASH_VARIANT_TYPE_STRING,
		CELL_FLASH_VARIANT_TYPE_OBJECT,
		CELL_FLASH_VARIANT_TYPE_ARRAY,
		CELL_FLASH_VARIANT_TYPE_ATOM,
		CELL_FLASH_VARIANT_TYPE_LIM_ = 0xffffffff // reserved: you don't use this value
	} CellFlashVariantType;

	typedef struct CellFlashVariant {
		CellFlashVariantType type;
		union {
			CellFlashBool boolValue;
			uint32_t intValue;
			double doubleValue;
			CellFlashString stringValue;
			CellFlashObject *objectValue; /// for NPAPI\. usually unused.
			CellFlashArray arrayValue; /// Array on ActionScript
			CellFlashAtom atomValue; /// Object/Class/Function on ActionSctipt @see CellFlashAtom.h
		} value;
	} CellFlashVariant;

	typedef enum CellFlashCircuit{
		CELL_FLASH_CIRCUIT_NETWORK = 1,
		CELL_FLASH_CIRCUIT_STORAGE = 2,
		CELL_FLASH_CIRCUIT_LIM_ = 0xffffffff // reserved: you don't use this value
	} CellFlashCircuit;


#define CELL_FLASH_KEYC_SHIFT (-1L)
#define CELL_FLASH_KEYC_CTRL (-2L)
#define CELL_FLASH_KEYC_ALT (-3L)
	
	
	
	//**********************************************************************
	//* host interface (called from libflash. handler functions)
	//**********************************************************************

	/**
	   @typedef typedef CellFlashObject* (*CellFlashCoreCreateObjectFunc)(CellFlashInstance* instance, CellFlashClass* aClass);
	   @param npp the player instance
	   @param aClass the prototype for the class
	   @retval 
	 */
	typedef CellFlashObject* (*CellFlashCoreCreateObjectFunc)(CellFlashInstance* instance, CellFlashClass* aClass);

	/**
	   @typedef typedef void (*CellFlashCoreReleaseObjectFunc)(CellFlashObject* obj);
	   
	   @param obj the object to be released
	 */
	typedef void (*CellFlashCoreReleaseObjectFunc)(CellFlashObject*);

	/**
	   @typedef typedef CellFlashBool (*CellFlashCoreEvaluateFunc)(CellFlashInstance* instance, CellFlashObject* obj, const char* func, int argc, CellFlashVariant* argv, CellFlashVariant* result);
	   the ramp for ExternalInterface.call().
	   calling ExternalInterface.call() in ActionScript, this callback should be called in same context.

	   @param npp the player instance
	   @param obj unused
	   @param func the name of Funcion
	   @param argc the number of arguments
	   @param argv the arguments. variant is accessible only in this context. (you can not modify them)
	   @param [out] result the result string. this should be released by CellFlashCoreReleaseVarT later.
	   @retval true if it successes
	 */
	typedef CellFlashBool (*CellFlashCoreEvaluateFunc)(CellFlashInstance* instance, CellFlashObject* obj, const char* func, int argc, CellFlashVariant* argv, CellFlashVariant* result);

	/**
	 @typedef typedef CellFlashBool (*CellFlashCoreInvokeFunc)(CellFlashInstance* instance, CellFlashObject* obj, CellFlashIdent methodname, const CellFlashVariant* args, uint32_t argc, CellFlashVariant* result);
	 the old school type ramp for ExternalInterface.call().
	 */
	typedef CellFlashBool (*CellFlashCoreInvokeFunc)(CellFlashInstance* instance, CellFlashObject* obj, CellFlashIdent methodname, CellFlashVariant* args, uint32_t argc, CellFlashVariant* result);

	/**
	   @typedef typedef CellFlashIdent (*CellFlashCoreGetStringIdentFunc)(const char* name);
	   the converter from string to ident.
	   CellFlashIdent should be the primitive that points the name of symbol instead of using the name.
	   usually it just only returns an 'name' as an ident.
	   @param name the name of symbol
	   @retval string
	   @see CellFlashCoreUtf8FromIdentFunc
	 */
	typedef CellFlashIdent (*CellFlashCoreGetStringIdentFunc)(const char* name);

	typedef CellFlashBool (*CellFlashCoreGetPropFunc)(CellFlashInstance* instance, CellFlashObject* obj, CellFlashIdent propname, CellFlashVariant* result);
	typedef CellFlashBool (*CellFlashCoreSetPropFunc)(CellFlashInstance* instance, CellFlashObject* obj, CellFlashIdent propname, const CellFlashVariant* result);


	/**
	   @typedef typedef void (*CellFlashCoreReleaseVarFunc)(CellFlashVariant* var);
	   an allocated object in the host-application will be released by this.
	   @param var the variant to be released.
	 */
	typedef void (*CellFlashCoreReleaseVarFunc)(CellFlashVariant* var);
	typedef CellFlashObject* (*CellFlashCoreRetainObjectFunc)(CellFlashObject* obj);
	typedef void (*CellFlashCoreSetExceptionFunc)(const CellFlashObject* obj, const char* msg);

	/**
	   @typedef typedef char* (*CellFlashCoreUtf8FromIdentFunc)(CellFlashIdent ident);
	   the converter from indet to string.
	   usually it just only returns an ident as the name.
	   @param ident the identity that points a name of symbol.
	   @retval string
	   @see CellFlashCoreGetStringIdentFunc
	 */
	typedef char* (*CellFlashCoreUtf8FromIdentFunc)(CellFlashIdent ident);

	typedef void (*CellFlashCoreInvalidRectFunc)(CellFlashInstance* instance, CellFlashRect* rect);
	typedef void (*CellFlashCoreInvalidateRegionFunc)(CellFlashInstance* instance, CellFlashRegion region);
	typedef void (*CellFlashCoreForceRedrawFunc)(CellFlashInstance* instance);

	/**
	   @typedef typedef CellFlashError (*CellFlashCoreSetValueFunc)(CellFlashInstance* instance, CellFlashVar var, void* value);
	   the setter to the host-application.
	   var is an identity for type of the 'variable'.
	   the variable is cond variable indicates the container's spec.
	   @param instance
	   @param var var is condition variable indicates the container's spec.
	   @param value 
	   @retval true if it successes.
	 */
	typedef CellFlashError (*CellFlashCoreSetValueFunc)(CellFlashInstance* instance, CellFlashVar var, const void* value);

	/**
	   @typedef typedef CellFlashError (*CellFlashCoreGetValueFunc)(CellFlashInstance* instance, CellFlashHostVar var, void* value);
	   the getter to the host-application.
	   var is an identity for type of the 'variable'.
	   the variable is condition variable indicates the host's condition.

	   @jp
	   @note 通常、何もせず true を返すだけで問題ありません。しかしこのコールバックはFlashプレイヤーのセキュリティ要件のためドメインチェックに使用されることに注意してください。特に外部リソースのリクエストや ExternalInterface の使用に際して、いくつかの var ("location", "href") について正しい応答をする必要があります。
	   @endjp

	   @en
	   @note 
	   usually, you do nothing and just only returns 'true'.
	   however, some cases want your correct response each conditions. for the flash player's security policies.
       mostly requesting the external uri or usage of the ExternalInterface.call(), you should be make a certainly response.
	   esp about "location" and "href". these two variables are used to identify SWF/requester domains matches.
	   @enden

	   @param instance
	   @param var var is condition variable indicates the host's spec.
	   @param [out] value 
	   @retval true if it successes. 
	 */
	typedef CellFlashError (*CellFlashCoreGetValueFunc)(CellFlashInstance* instance, CellFlashHostVar var, void* value);

	typedef void (*CellFlashCoreStatusFunc)(CellFlashInstance* instance, const char* msg);
	typedef const char* (*CellFlashCoreUaFunc)(CellFlashInstance* instance);
	typedef void (*CellFlashCoreReloadPluginFunc)(CellFlashBool reloadPages);

	typedef void (*CellFlashCoreChangeCircuitStatusFunc)(CellFlashCircuit circuit, CellFlashBool up);
	typedef void (*CellFlashCoreDebugPrintFunc)(int level, const char* msg);

	typedef struct CellFlashCoreInterface {
		CellFlashCoreCreateObjectFunc createObjectFunc;
		CellFlashCoreReleaseObjectFunc releaseObjectFunc;
		CellFlashCoreEvaluateFunc evaluateFunc;
		CellFlashCoreGetPropFunc getPropFunc;
		CellFlashCoreGetStringIdentFunc getStringIdentFunc;
		CellFlashCoreSetPropFunc setPropFunc;
		CellFlashCoreInvokeFunc invokeFunc;
		CellFlashCoreReleaseVarFunc releaseVarFunc;
		CellFlashCoreRetainObjectFunc retainObjectFunc;
		CellFlashCoreSetExceptionFunc setExceptionFunc;
		CellFlashCoreUtf8FromIdentFunc  utf8FromIdentFunc;

		CellFlashCoreInvalidRectFunc invalidRectFunc;
		CellFlashCoreInvalidateRegionFunc invalidateRegionFunc;
		CellFlashCoreForceRedrawFunc forceRedrawFunc;

		CellFlashCoreSetValueFunc setValueFunc;
		CellFlashCoreGetValueFunc getValueFunc;
		CellFlashCoreStatusFunc statusFunc;
		CellFlashCoreUaFunc uaFunc;
		CellFlashCoreReloadPluginFunc reloadPluginFunc;

		CellFlashCoreChangeCircuitStatusFunc changeCircuitStatusFunc;
		CellFlashCoreDebugPrintFunc debugPrintFunc;

	} CellFlashCoreInterface;


	typedef void* (*CellFlashFileOpenFunc)(const char* path, const char* mode, CellFlashBool temp);
	typedef void (*CellFlashFileCloseFunc)(void* handle);
	typedef int32_t (*CellFlashFileTellFunc)(void* handle);
	typedef int32_t (*CellFlashFileSeekFunc)(void* handle, uint32_t curpos, uint32_t mode);
	typedef int32_t (*CellFlashFileFlushFunc)(void* handle);
	typedef size_t (*CellFlashFileWriteFunc)(const void* buf, size_t count, size_t size, void* handle);
	typedef size_t (*CellFlashFileReadFunc)(void* buf, size_t count, size_t size, void* handle);
	typedef char* (*CellFlashFileGetsFunc)(void* buf, size_t size, void* handle);
	typedef int32_t (*CellFlashFileUnlinkFunc)(const char* path);
	typedef CellFlashBool (*CellFlashFileIsMappedFileFunc)(const char* path);

	typedef struct CellFlashFileInterface {
		CellFlashFileOpenFunc fopenFunc;
		CellFlashFileCloseFunc fcloseFunc;
		CellFlashFileTellFunc ftellFunc;
		CellFlashFileSeekFunc fseekFunc;
		CellFlashFileFlushFunc fflushFunc;
		CellFlashFileWriteFunc fwriteFunc;
		CellFlashFileReadFunc freadFunc;
		CellFlashFileGetsFunc fgetsFunc;
		CellFlashFileUnlinkFunc unlinkFunc;
		CellFlashFileIsMappedFileFunc isMappedFileFunc;
	} CellFlashFileInterface;


	/**
	   @typedef typedef void* (*CellFlashMemoryAllocFunc)(uint32_t size);
	   memory allocator(allocate).
	   you have to implement.
	 */
	typedef void* (*CellFlashMemoryAllocFunc)(uint32_t size);
	/**
	   @typedef typedef void* (*CellFlashMemoryFreeFunc)(uint32_t size);
	   memory allocator(free).
	   you have to implement.
	 */
	typedef void (*CellFlashMemoryFreeFunc)(void* ptr);

	typedef struct CellFlashMemoryInterface {
		CellFlashMemoryAllocFunc memAllocFunc;
		CellFlashMemoryFreeFunc memFreeFunc;
	} CellFlashMemoryInterface ;


	typedef void* (*CellFlashSoundOpenPortFunc)(uint32_t ch, uint32_t sampling, uint32_t bytes, CellFlashSoundSampleType type);
	typedef int (*CellFlashSoundSendPcmFunc)(void* port, void* buf, uint32_t len, uint32_t samplenum, uint32_t ch, uint32_t samplingrate, CellFlashSoundSampleType type);
	typedef void (*CellFlashSoundClosePortFunc)(void* port);

	typedef struct CellFlashSoundInterface {
		CellFlashSoundOpenPortFunc openPort;
		CellFlashSoundSendPcmFunc sendPcm;
		CellFlashSoundClosePortFunc closePort;
	} CellFlashSoundInterface;


	typedef CellFlashError (*CellFlashStreamGetUrlFunc)(CellFlashInstance* instance, const char* url, const char* win);
	typedef CellFlashError (*CellFlashStreamPostUrlFunc)(CellFlashInstance* instance, const char* url, const char* target, uint32_t len, const char* buf, CellFlashBool file);
	typedef CellFlashError (*CellFlashStreamGetUrlNotifyFunc)(CellFlashInstance* instance, const char* url, const char* window, void* notifydata);
	typedef CellFlashError (*CellFlashStreamPostUrlNotifyFunc)(CellFlashInstance* instance, const char* url, const char* window, uint32_t len, const char* buf, CellFlashBool file, void* notifydata);
	typedef CellFlashError (*CellFlashStreamRequestReadFunc)(CellFlashInstance* instance, CellFlashStream stream, CellFlashRange* range);
	typedef CellFlashError (*CellFlashStreamNewStreamFunc)(CellFlashInstance* instance, CellFlashMimetype type, const char* window, CellFlashStream* streamPtr);
	typedef int32_t (*CellFlashStreamWriteFunc)(CellFlashInstance* instance, CellFlashStream stream, int32_t len, void* buf);
	typedef CellFlashError (*CellFlashStreamDestroyStreamFunc)(CellFlashInstance* instance, CellFlashStream* stream, CellFlashError reason);

	typedef struct CellFlashStreamInterface {
		CellFlashStreamGetUrlFunc getUrlFunc;
		CellFlashStreamPostUrlFunc postUrlFunc;
		CellFlashStreamGetUrlNotifyFunc getUrlNotifyFunc;
		CellFlashStreamPostUrlNotifyFunc postUrlNotifyFunc;
		CellFlashStreamRequestReadFunc requestReadFunc;
		CellFlashStreamNewStreamFunc newStreamFunc;
		CellFlashStreamWriteFunc writeFunc;
		CellFlashStreamDestroyStreamFunc destroyStreamFunc;
	} CellFlashStreamInterface ;









	//**********************************************************************
	//* container interface (called from host application to control flash) 
	//**********************************************************************
	CellFlashError cellFlashInitialize(	const CellFlashCoreParamset* const param,
										const CellFlashCoreInterface* const coreInterface,
										const CellFlashMemoryInterface* const memoryInterface,
										const CellFlashSoundInterface* const soundInterface,
										const CellFlashStreamInterface* const streamInterface,
										const CellFlashFileInterface* const fileInterface);

	CellFlashError cellFlashShutdown();

	/** 
	 *  @fn 	CellFlashInstance** cellFlashCreate(const CellFlashParamsetT* const, void*);
	 *  @jp
	 *  @param パラメータセットのインスタンスへのポインタ
	 *  @param 任意のデータ
	 *  @retval プレイヤーインスタンス。失敗した場合はNULL
	 *
	 *  @brief プレイヤーインスタンスの生成
	 *  @note プレイヤーインスタンスを生成します。Flashプレイヤーの初期化が完了してある必要があります。 @NOMTSAFE
	 *  @endjp
	 *
	 *  @en
	 *  @param a pointer to a description of config for the player
	 *  @param userdata
	 *  @retval an instance of the player. if fail, returns NULL.
	 *
	 *  @brief creating an instance of the player 
	 *  @note need to be finished cellFlashInitialize() @NOMTSAFE
	 *  @enden
	 */
	CellFlashInstance* cellFlashCreate(const CellFlashParamset* const param, void* pObj);

	/**
	 *  @fn 	int cellFlashDestroy(CellFlashInstance*);
	 *  @jp
	 *  @param パラメータセットのインスタンスへのポインタ
	 *  @retval パラメータがNULLの場合 CellFlashErrorInval を返します
	 *
	 *  @brief プレイヤーインスタンスの破壊
	 *  @note プレイヤーインスタンスを破壊します @NOMTSAFE
	 *  @endjp
	 *
	 *  @en
	 *  @param an instance of the player
	 *  @retval if param is null, returns CellFlashErrorInval.
	 *
	 *  @brief destroying an instance of the player
	 *  @note @NOMTSAFE
	 *  @enden
	 */
	CellFlashError cellFlashDestroy(CellFlashInstance* instance);


	CellFlashBool cellFlashLoadMovie(CellFlashInstance* instance, const char* const name, const char* const baseurl, const char* const url);



	void cellFlashStart(CellFlashInstance* instance);
	void cellFlashStop(CellFlashInstance* instance);
	void cellFlashResume(CellFlashInstance* instance);
	void cellFlashPause(CellFlashInstance* instance);


	/**
	 *  @fn 	CellFlashError cellFlashDoplay(CellFlashInstance*, CellFlashBool);
	 *  step a player's frame. you should periodically invoke this for each players , to carry it on.
	 *  @param pid the player
	 *  @param socket. true if player polls sockets. usually this will be true.
	 */
	CellFlashError cellFlashDoPlay(CellFlashInstance* instance, CellFlashBool bSocket);

	void cellFlashStreamProc(CellFlashInstance* instance);

	void cellFlashDraw(const CellFlashInstance* instance, uint8_t* const buf);
	void cellFlashDrawPitch(const CellFlashInstance* instance, uint8_t* const buf, uint32_t pitch);
	void cellFlashSubDraw(const CellFlashInstance* instance, uint8_t* const buf, uint32_t pitch, uint32_t x, uint32_t y, uint32_t w, uint32_t h);


	/**
	 * @fn 	CellFlashError cellFlashCall(CellFlashInstance*, const char* methodname, int argc, CellFlashVariant* args, CellFlashVariant* retval);
	 * @jp
	 * ActionScript コールバック関数の呼び出し
	 * AS3 の ExternalInterface.addCallback()  によって登録したコールバック関数へネイティブから呼び出します。
	 * 実行はこのコンテキストで即座に行われます。終了まで戻らないことに注意してください。 @NOMTSAFE
	 * @param プレイヤーインスタンス
	 * @param ExternalInterface.addCallback() で登録したASの関数名
	 * @param 引数の数
	 * @param 引数の配列
	 * @param コールバック関数の戻り値
	 * @retval エラーコード
	 * @endjp
	 *
	 * @en
	 * calling ActionScript's function.
	 * calling this function, callback function on ActionScript runs. the callback function is the registee by ExternalInterface.addCallback() (AS3).
	 * this execution is done on time and program counter does not return until it ends.
	 * @param an instance of the player
	 * @param the function name was registed by ExternalInterface.addCallback()
	 * @param a num of arguments
	 * @param a vector of arguments
	 * @param a result of callback
	 * @retval errocode
	 * @enden
	 */
	CellFlashError cellFlashCall(CellFlashInstance* instance, const char* methodname, int argc, CellFlashVariant* args, CellFlashVariant* retval);

	/**
	 * @fn void cellFlashDeleteVariant(CellFlashInstance* pid, CellFlashVariant* v);
	 * delete CellFlashVariant that be allocated by library.
	 * @see CellFlashCall, CellFlashAtomCallprop, CellFlashAtomGetprop
	 */
	void cellFlashDeleteVariant(CellFlashInstance* instance, CellFlashVariant* v);



	/**
	 * @fn CellFlashError cellFlashSetPointerPos(CellFlashInstance* pid, int x, int y);
	 * set a position of pointing without changing the button state. 
	 * if the button state does not change, it's better than CellFlashSetPtState(x, y, false).
	 * @param pid the player
	 * @param x from left in the screen coordinate.
	 * @param y from top in the screen coordinate.
	 */
	CellFlashError cellFlashSetPointerPos(CellFlashInstance* instance, int x, int y);

	/**
	 * @fn CellFlashError cellFlashSetPointerState(CellFlashInstance* pid, int x, int y, CellFlashBool press);
	 * set a position of pointing with changing the button state. 
	 * @param pid the player
	 * @param x from left in the screen coordinate.
	 * @param y from top in the screen coordinate.
	 * @param button state. true says 'pressed', false says 'released'.
	 */
	CellFlashError cellFlashSetPointerState(CellFlashInstance* instance, int x, int y, CellFlashBool bPressed);
	CellFlashError cellFlashGetPointerState(CellFlashInstance* instance, int* x, int* y, CellFlashCursor* cursor);
	CellFlashError cellFlashSetKeyState(CellFlashInstance* instance, CellFlashBool bDown, int code, int reserved0, int reserved1, int reserved2);
	CellFlashError cellFlashSetKeyStateRaw(CellFlashInstance* instance, uint32_t arrange,  CellFlashBool bDown, int32_t code, uint32_t ledstate);


	/**
	 *  @fn 	void cellFlashMute(CellFlashInstance*, CellFlashBool mute);
	 *  if 'mute' is true, the player is muted. or not be unmuted.
	 */
	void cellFlashSetMute(CellFlashInstance* instance, CellFlashBool bMute);
	CellFlashBool cellFlashIsMuted(CellFlashInstance* instance);


	CellFlashError cellFlashGetUsedHeap(CellFlashInstance* instance, size_t* size);
	/**
	 *  @fn 	CellFlashError cellFlashGetFrames(CellFlashInstance*, int* total, int* current);
	 *  get a frame state. but it is un-useable.
	 */
	CellFlashError cellFlashGetFrames(CellFlashInstance* instance, int* total, int* current);
	/**
	 *  @fn 	CellFlashError CellFlashGetStageSize(CellFlashInstance*, int* width, int* height);
	 *  get the original stage size. the original stage size is embeded in swf.
	 */
	CellFlashError cellFlashGetStageSize(CellFlashInstance* instance, int* width, int* height);

	float cellFlashGetDelay(CellFlashInstance* instance);
	CellFlashBool cellFlashIsSyncDownloading(CellFlashInstance*  instance);

	CellFlashError cellFlashGetSurfaceInfo(const CellFlashInstance* instance, uint8_t** ptr, int* pitch, int* width, int* height);



#if defined(__cplusplus)
}
#endif

#endif
