/*****************************************************************
 * SCE CONFIDENTIAL
 PlayStation(R)3 Programmer Tool Runtime Library 475.001
 * Copyright (c) 2010 Sony Computer Entertainment Inc.
 * All Rights Reserved.
 *
 * R & D Division, Sony Computer Entertainment Inc.
 *****************************************************************/

#ifndef __CELL_VOICE_TYPES_H__
#define __CELL_VOICE_TYPES_H__


#if defined(__LANGUAGE_C_PLUS_PLUS)||defined(__cplusplus)||defined(c_plusplus)
extern "C" {
#endif

#include <sys/types.h>

/// The maximum number of active IN-VOICE sessions in system memory mode.
#define CELLVOICE_MAX_IN_VOICE_PORT 32  
/// The maximum number of active OUT-VOICE sessions in system memory mode. 
#define CELLVOICE_MAX_OUT_VOICE_PORT  4 
/// The maximum number of active IN-VOICE sessions in 1MB game memory mode.
#define CELLVOICE_GAME_1MB_MAX_IN_VOICE_PORT 8  
/// The maximum number of active OUT-VOICE sessions in 1MB game memory mode. 
#define CELLVOICE_GAME_1MB_MAX_OUT_VOICE_PORT  2 
/// The maximum number of ports supported. Constrained by memory limitations.
#define CELLVOICE_MAX_PORT  128       
/// The invalid port ID.
#define CELLVOICE_INVALID_PORT_ID  0xff       

/* error code */
/*
	CELL_ERROR_FACILITY_SOUND			0x031
	libvoice							0x8031_0801 - 0x8031_08ff
 */
/// The libvoice library has not been initialized.
#define CELL_VOICE_ERROR_LIBVOICE_NOT_INIT 		CELL_ERROR_CAST(0x80310801)
/// The libvoice library has already been initialized.
#define CELL_VOICE_ERROR_LIBVOICE_INITIALIZED	CELL_ERROR_CAST(0x80310802)
/// Errors that do not belong to any other category.
#define CELL_VOICE_ERROR_GENERAL 				CELL_ERROR_CAST(0x80310803)
/// Invalid port.
#define CELL_VOICE_ERROR_PORT_INVALID			CELL_ERROR_CAST(0x80310804)
/// Invalid argument.
#define	CELL_VOICE_ERROR_ARGUMENT_INVALID		CELL_ERROR_CAST(0x80310805)
/// Invalid container.
#define	CELL_VOICE_ERROR_CONTAINER_INVALID		CELL_ERROR_CAST(0x80310806)
/// Topology error.
#define CELL_VOICE_ERROR_TOPOLOGY				CELL_ERROR_CAST(0x80310807)
/// Insufficient resources.
#define CELL_VOICE_ERROR_RESOURCE_INSUFFICIENT	CELL_ERROR_CAST(0x80310808)
/// Feature not yet implemented. 
#define CELL_VOICE_ERROR_NOT_IMPLEMENTED		CELL_ERROR_CAST(0x80310809)
/// Invalid address.
#define CELL_VOICE_ERROR_ADDRESS_INVALID		CELL_ERROR_CAST(0x8031080a)
/// System Voice Service is detached. 
#define CELL_VOICE_ERROR_SERVICE_DETACHED		CELL_ERROR_CAST(0x8031080b)
/// System Voice Service is attached. 
#define CELL_VOICE_ERROR_SERVICE_ATTACHED		CELL_ERROR_CAST(0x8031080c)
/// System Voice Service not found. 
#define	CELL_VOICE_ERROR_SERVICE_NOT_FOUND	    CELL_ERROR_CAST(0x8031080d)
/// Shared memory error.
#define	CELL_VOICE_ERROR_SHAREDMEMORY			CELL_ERROR_CAST(0x8031080e)
/// Event Queue error.
#define	CELL_VOICE_ERROR_EVENT_QUEUE			CELL_ERROR_CAST(0x8031080f)
/// Invalid System Voice Service Handler.
#define	CELL_VOICE_ERROR_SERVICE_HANDLE			CELL_ERROR_CAST(0x80310810)
/// Voice event dispatch fails.
#define	CELL_VOICE_ERROR_EVENT_DISPATCH			CELL_ERROR_CAST(0x80310811)
/// Device was not present.
#define	CELL_VOICE_ERROR_DEVICE_NOT_PRESENT		CELL_ERROR_CAST(0x80310812)


/// Port types for libvoice.
/// @see <c>CellVoiceBasePortInfo</c>, <c>CellVoicePortParam</c>
typedef enum _CellVoicePortType
{
	/// Null port
    CELLVOICE_PORTTYPE_NULL = -1,
    /// Microphone input stream port
    CELLVOICE_PORTTYPE_IN_MIC = 0,
    /// PCM input stream port
    CELLVOICE_PORTTYPE_IN_PCMAUDIO = 1,
    /// Voice input stream port
    CELLVOICE_PORTTYPE_IN_VOICE = 2,
    /// PCM output stream port
    CELLVOICE_PORTTYPE_OUT_PCMAUDIO = 3,
    /// Voice output stream port
    CELLVOICE_PORTTYPE_OUT_VOICE = 4,
    /// Audio output stream port
    CELLVOICE_PORTTYPE_OUT_SECONDARY = 5,
} CellVoicePortType;

/// Enumeration for the state of a port. 
/// Each port has its own state. 
/// State changes can be the result of events generated by System Voice Service,
/// of libvoice function calls made directly by the application, or of the 
/// data level of the buffer.
///
/// The following diagram shows how the states change with 
/// specific actions or events on the port:
///
/// @code
/// NULL ---------------- Topology Build Completed -----------------> IDLE
/// IDLE -------------------- cellVoiceStart -----------------------> READY
/// READY ------------ CELLVOICE_EVENT_SERVICE_ATTACHED ------------> BUFFERING
/// BUFFERING ------------ Data reaches threshold ------------------> RUNNING
/// RUNNING ---- CELLVOICE_EVENT_SERVICE_DETACHED/cellVoiceStop ----> IDLE
/// READY ------------------- cellVoiceStop ------------------------> IDLE
/// @endcode   
/// @see <c>CellVoiceBasePortInfo</c>, <c>CellVoiceEventType</c>, <c>cellVoiceStart</c>, <c>cellVoiceStop</c>
typedef enum _CellVoicePortState
{
    CELLVOICE_PORTSTATE_NULL = -1,      /**< The initial port state */
    CELLVOICE_PORTSTATE_IDLE = 0,       /**< The port is idle*/
    CELLVOICE_PORTSTATE_READY = 1,      /**< The port is ready */
    CELLVOICE_PORTSTATE_BUFFERING = 2,	/**< The port is buffering data */
    CELLVOICE_PORTSTATE_RUNNING = 3,    /**< The port is running */
} CellVoicePortState;

/// Enumeration for the bit rate for a port.
/// @see <c>CellVoicePortParam</c>, <c>cellVoiceSetBitRate</c>
typedef enum _CellVoiceBitRate
{
    CELLVOICE_BITRATE_NULL  = -1,          /**< Null bit rate */  
    CELLVOICE_BITRATE_3850  = 3850,        /**< Bit rate of 3850 bps */ 
    CELLVOICE_BITRATE_4650  = 4650,        /**< Bit rate of 4650 bps */ 
    CELLVOICE_BITRATE_5700  = 5700,        /**< Bit rate of 5700 bps */ 
    CELLVOICE_BITRATE_7300  = 7300,        /**< Bit rate of 7300 bps */
    CELLVOICE_BITRATE_14400 = 14400,       /**< Bit rate of 14400 bps */
    CELLVOICE_BITRATE_16000 = 16000,       /**< Bit rate of 16000 bps */
    CELLVOICE_BITRATE_22533 = 22533        /**< Bit rate of 22533 bps */
} CellVoiceBitRate;

/// Enumeration for the sampling rate for a port.
/// @see <c>CellVoicePCMFormat</c>, <c>cellVoiceCreatePort</c>
typedef enum _CellVoiceSamplingRate
{
    CELLVOICE_SAMPLINGRATE_NULL = -1,   	/**< Null sampling rate */  
    CELLVOICE_SAMPLINGRATE_16000 = 16000,   /**< 16000 sampling rate */  
} CellVoiceSamplingRate;


/// Enumeration for libvoice port event types.
/// These event types are used with the <c>sys_event_t</c> event structure, defined as follows: 
///
/// <table>
/// <tr><td><b>Data Member</b></td><td><b>Description</b></td></tr>
/// <tr><td><c><i>source</i></c></td><td>The source port name, the name registered 
///                             at <c>cellVoiceSetNotifyEventQueue()</c>.</td></tr>
/// <tr><td><c><i>data1</i></c></td><td>The event type.</td></tr>
/// <tr><td><c><i>data2</i></c></td><td>The specified port ID to which the event is related. 
/// <tr><td><c><i>data3</i></c></td><td>Undefined.</td></tr> 
///	</table>
///
/// The application can handle <c>CELLVOICE_EVENT_PORT_ATTACHED</c> and <c>CELLVOICE_EVENT_PORT_DETACHED</c>
/// when the System Voice Service is handling the microphone internally.  
/// The applicatioin can handle <c>CELLVOICE_EVENT_PORT_WEAK_ATTACHED</c> and <c>CELLVOICE_EVENT_PORT_WEAK_DETACHED</c>
/// when applicatioin uses libmic directly in addition to libvoice. 
/// @see <c>CellVoiceInitParam</c>, <c>sys_event_t</c> (see the <i>Cell OS Lv-2 System Call and Library Reference</i>)
typedef enum _CellVoiceEventType
{
    CELLVOICE_EVENT_DATA_ERROR = 1 << 0,            /**< Any error related to audio devices (microphone or secondary) */
    CELLVOICE_EVENT_PORT_ATTACHED = 1 << 1,         /**< Microphone is attached */ 
    CELLVOICE_EVENT_PORT_DETACHED = 1 << 2,         /**< Microphone is detached */
    CELLVOICE_EVENT_SERVICE_ATTACHED = 1 << 3,      /**< System Voice Service is attached */
    CELLVOICE_EVENT_SERVICE_DETACHED = 1 << 4,       /**< System Voice Service is detached */
    CELLVOICE_EVENT_PORT_WEAK_ATTACHED = 1 << 5,     /**< Microphone is attached to libvoice because libmic releases microphon by cellMicClose */ 
    CELLVOICE_EVENT_PORT_WEAK_DETACHED = 1 << 6,     /**< Microphone is taken over by libmic open calls(cellMicOpen, cellMicOpenEx, cellMicOpenRaw), and is detached from libvoice */
	} CellVoiceEventType;

/// Enumeration for the PCM data type.
/// @see <c>CellVoicePCMFormat</c>
typedef enum _CellVoicePcmDataType
{
	CELLVOICE_PCM_NULL = -1,                        /**< Null data type */
    CELLVOICE_PCM_FLOAT = 0,                        /**< 32-bit Float */
    CELLVOICE_PCM_FLOAT_LITTLE_ENDIAN   = 1,        /**< 32-bit Float stored in little endian format */
    CELLVOICE_PCM_SHORT = 2,                        /**< 16-bit Short */
    CELLVOICE_PCM_SHORT_LITTLE_ENDIAN   = 3,        /**< 16-bit Short stored in little endian format */
    CELLVOICE_PCM_INTEGER   = 4,                    /**< 32-bit Integer */
    CELLVOICE_PCM_INTEGER_LITTLE_ENDIAN = 5,        /**< 32-bit Integer stored in little endian format */
} CellVoicePcmDataType;


/// Enumeration indicating the libvoice data structure versions. There will be additional versions added for future libvoice data structure extensions.
/// 
typedef enum _CellVoiceVersionCheck
{
    CELLVOICE_VERSION_100  = 100,            /**< Version 100 */
} CellVoiceVersionCheck;

/// Data structure for the PCM format.
/// @see <c>CellVoicePortParam</c> 
typedef struct _CellVoicePCMFormat
{
    /// Number of channels. Unused for now
    uint8_t numChannels;	
    /// Alignment. For example, 4-byte aligned or 2-byte aligned. Unused for now
    uint8_t sampleAlignment;
    /// PCM data type. 
    CellVoicePcmDataType dataType;
    /// Sampling rate.
    CellVoiceSamplingRate sampleRate;
} CellVoicePCMFormat;

/// Base port information. 
/// This structure holds the topology for the port returned from <c>cellVoiceGetPortInfo()</c>.
/// @see <c>cellVoiceGetPortInfo</c> 
//  @see <c>cellVoiceDebugTopology</c>
typedef struct _CellVoiceBasePortInfo
{
    /// Port type.
    CellVoicePortType portType; 
    /// Port state. 
    CellVoicePortState state;
    /// Number of connections.
    uint16_t numEdge;
    /// Connected ports.
    uint32_t *pEdge;
    /// When <c><i>portType</i></c> is an OPort: The number of bytes that can be removed from the buffer.
	/// When <c><i>portType</i></c> is an IPort: The number of bytes that can be inserted to the buffer.
    uint32_t numByte;
	/// Unit frame size.
    uint32_t frameSize;
}  CellVoiceBasePortInfo __attribute__ ((aligned(64)));

/// Enumeration for port attributes.
/// Use <c>cellVoiceGetPortAttr()</c> to request both the calculated and user-set 
/// (enabled or disabled) values for the attributes.
/// @see <c>cellVoiceGetPortAttr</c>, <c>cellVoiceSetPortAttr</c>, <c>CellVoicePortParam</c>
typedef enum _CellVoicePortAttr
{
    /// Denotes the energy level attribute. This attribute can be either true or false.
	/// It is applicable for all ports. 
    /// When the attribute is read, the get function returns a float value between 0.0 and 100.0.
    CELLVOICE_ATTR_ENERGY_LEVEL = 1000,
    /// Denotes the VAD attribute. This attribute has an integer value from 0 to 10.
    /// It is only applicable for <c>CELLVOICE_PORTTYPE_IN_MIC</c>. Any value greater than 10 or equal to 0 
    /// causes the default value for the device type to be set. Refer to the <i>libmic Reference</i>
    /// for the default values.
    CELLVOICE_ATTR_VAD = 1001,
    /// Denotes the discontinuous transmit attribute, which can be enabled or disabled when VAD has 
    /// been enabled. DTX mode is enabled by default. Its value is either true or false.
    /// When discontinuous transmit mode is enabled, data is not produced when there is no voice
    /// activity. Disabling DTX causes data to be produced, even if there is no voice activity.
    /// This attribute is only applicable for <c>CELLVOICE_PORTTYPE_IN_MIC</c>.
	/// <c>CELL_VOICE_ERROR_ARGUMENT_INVALID</c> is returned if the attribute is used for other port types.
    CELLVOICE_ATTR_DTX = 1002,
    /// Denotes the auto resampling attribute. Disabled by default. 
    /// Its value is either true or false. When true, 
    /// automatic resampling is enabled. When false, automatic resampling is disabled. 
    /// Auto resample attempts to keep the port input buffer level to the same level as 
    /// the buffer threshold, specified with <c>CellVoicePortParam</c>. 
    /// This setting is only applicable to a compressed input port, <c>CELLVOICE_PORTTYPE_IN_VOICE</c>.
	/// <c>CELL_VOICE_ERROR_ARGUMENT_INVALID</c> is returned if the attribute is sent to the set function for other port types.
    /// The <c>cellVoiceGetPortAttr()</c> function is not applicable for this attribute.
	/// <c>CELL_VOICE_ERROR_ARGUMENT_INVALID</c> is returned if the attribute is passed to the get attribute function.
	CELLVOICE_ATTR_AUTO_RESAMPLE = 1003,
    /// Denotes the estimated average latency attribute. Its value is a float of the estimated latency in ms. This
    /// attributes applies to all port types. The <c>cellVoiceSetPortAttr()</c> function is not applicable for this attribute.
	/// <c>CELL_VOICE_ERROR_ARGUMENT_INVALID</c> is returned if the attribute is passed to the set attribute function.
    CELLVOICE_ATTR_LATENCY = 1004,
    /// Denotes the silence threshold attribute. This attribute has an integer value in milliseconds.
	/// Using this attribute, you can set the desired interval of silence after which no further 
	/// data will be sent. When the attribute is read, the get function returns the current silence time. The default
	/// value of the silence threshold is 300 milliseconds.
    /// This attribute is only applicable to <c>CELLVOICE_PORTTYPE_IN_MIC</c>. 
	/// <c>CELL_VOICE_ERROR_ARGUMENT_INVALID</c> is returned if the attribute is used for other port types.
    CELLVOICE_ATTR_SILENCE_THRESHOLD = 1005,

} CellVoicePortAttr;

/// Port parameters. This structure is used to create and update ports. 
/// @see <c>cellVoiceCreatePort</c>, <c>cellVoiceUpdatePort</c>
typedef struct _CellVoicePortParam
{
    /// Port type
    CellVoicePortType portType;
    /// Shared buffer threshold (in millisecond). 
	/// <c><i>threshold</i></c> is not applicable to a microphone or output port.
	/// Maximum threshold value for IPort:<br>
	/// For <c>CELLVOICE_PORTTYPE_IN_VOICE</c>:
	/// <ul>
	///  <li>Bit rate  3850: 4000 ms</li>
	///  <li>Bit rate  4560: 3300 ms</li>
	///  <li>Bit rate  5700: 2600 ms</li>
	///  <li>Bit rate  7300: 2000 ms</li>
	///  <li>Bit rate 14400: 1100 ms</li>
	///  <li>Bit rate 16000: 1000 ms</li>
	///  <li>Bit rate 22300:  700 ms</li>
	/// </ul>
	/// For <c>CELLVOICE_PORTTYPE_IN_PCMAUDIO</c>:
	/// <ul>
	/// <li>32-bit integer or float: (PCM-buffer-size/1024*16) ms</li>
	/// <li>16-bit integer: (PCM-buffer-size/512*16) ms</li>
	/// </ul>
    uint16_t threshold;
	/// Indicates whether the port is muted
    uint16_t bMute;
	/// Provides the port volume, a value of 0 silences the port. 
    float volume;
	// Union for CellVoicePortParam
    union  
    {
        struct 
        {
            CellVoiceBitRate bitrate;	/**< Bit rate for the port */
        } voice;					    /**< Voice information data structure */

        struct 	
        {
            uint32_t bufSize;			/**< Buffer size */
            CellVoicePCMFormat format;	/**< PCM format */
        } pcmaudio;					    /**< PCM audio information data structure */


        struct  	
        {
            uint32_t playerId;		   /**< Player ID */
        } device;					   /**< Device information data structure */
    } ; 				  /**< Union for CellVoicePortParam */
} CellVoicePortParam __attribute__ ((aligned(64)));


/// Enumeration for application type. System Voice Service supports two Game Chat configurations.
///
/// To use system memory mode, your application must use <c>cellVoiceInit()</c> and <c>cellVoiceStart()</c> to 
/// start System Voice Service. 
/// System memory mode: 
/// <ul>
/// <li>Does not use game space memory.</li> 
/// <li>Supports 32 decoders, four encoders, and PCM ports up to 512K.</li>
/// <li>Does not coexist with the system Background Music feature.</li> 
/// </ul>
///
/// To use 1MB game memory mode, your application must pass <c>CELLVOICE_APPTYPE_GAME_1MB</c> 
/// to <c>cellVoiceInitEx()</c>, and use <c>cellVoiceStartEx()</c> to start System Voice Service.	1MB game memory mode:
/// <ul> 
/// <li>Uses game space memory. The application provides 1MB memory for System Voice Service.</li> 
/// <li>Supports eight decoders, two encoders and PCM ports up to 80K</li>
/// <li>Coexists with the system Background Music feature.</li>
/// </ul>
typedef enum _CellVoiceAppType
{
	/// 1MB game memory mode.
	CELLVOICE_APPTYPE_GAME_1MB = 1 << 29,
} CellVoiceAppType;

/// Data structure for initializing libvoice. 
/// @see <c>cellVoiceInit</c>, <c>cellVoiceInitEx</c>
typedef struct _CellVoiceInitParam
{
    /// Unused. 
    CellVoiceEventType eventMask;
	// The following fields are added at CELLVOICE_VERSION_100. 
	/// The libvoice data structure version.
	CellVoiceVersionCheck version;
    /// Application configuration for System Voice Service.
	int32_t appType;
	/// Reserved bytes.
	uint8_t	reserved[32 - sizeof(int32_t) * 3];
} CellVoiceInitParam  __attribute__ ((aligned(32))); 

/// Data structure for providing a game space container to <c>cellVoiceStartEx()</c>. 
/// @see <c>cellVoiceStartEx</c>
typedef struct _CellVoiceStartParam
{
	// The following fields are added at CELLVOICE_VERSION_100. 
	// (Version checking for future extension.)
    /// Memory container. 
    sys_memory_container_t container;
    /// Reserved bytes. 
	uint8_t	reserved[32 - sizeof(int32_t) * 1];
}  CellVoiceStartParam __attribute__ ((aligned(32)));


#if defined(__LANGUAGE_C_PLUS_PLUS)||defined(__cplusplus)||defined(c_plusplus)
}

#endif
#endif // __CELL_VOICE_TYPES_H__
