/* SCE CONFIDENTIAL
 * PlayStation(R)3 Programmer Tool Runtime Library 475.001
 * Copyright (C) 2010 Sony Computer Entertainment Inc.
 * All Rights Reserved.
 */

#ifndef __CELL_SPURS_JOBQUEUE_H__
#define __CELL_SPURS_JOBQUEUE_H__

/* standard C++ header */
#include <stdint.h>
#include <stdbool.h>
#include <stddef.h> /* size_t */

#include <assert.h>
#include <ppu_intrinsics.h>

#include <cell/sysmodule.h>

/* spurs */
#include <cell/spurs/types.h>
#include <cell/spurs/job_descriptor.h>

/* job queue */
#include <cell/spurs/job_queue_define.h>
#include <cell/spurs/job_queue_types.h>
#include <cell/spurs/version.h>

#ifdef __cplusplus
extern "C" {
#endif
	/* APIs for CellSpursJobQueue */

	/* functions to set a parameter to CellSpursJobQueueAttribute */
	int cellSpursJobQueueAttributeInitialize(CellSpursJobQueueAttribute *attr);

	int cellSpursJobQueueAttributeSetMaxGrab(CellSpursJobQueueAttribute *attr, unsigned int maxGrab);
	int cellSpursJobQueueAttributeSetSubmitWithEntryLock(CellSpursJobQueueAttribute *attr, bool submitWithEntryLock);
	int cellSpursJobQueueAttributeSetDoBusyWaiting(CellSpursJobQueueAttribute *attr, bool doBusyWaiting);
	int cellSpursJobQueueAttributeSetIsHaltOnError(CellSpursJobQueueAttribute *attr, bool isHaltOnError);
	int cellSpursJobQueueAttributeSetIsJobTypeMemoryCheck(CellSpursJobQueueAttribute *attr, bool isJobTypeMemoryCheck);
	int cellSpursJobQueueAttributeSetMaxSizeJobDescriptor(CellSpursJobQueueAttribute *attr, unsigned int maxSizeJobDescriptor);
	//int cellSpursJobQueueAttributeSetMaxNumJobsOnSpu(CellSpursJobQueueAttribute *attr, unsigned int maxNumJobsOnSpu);
	int cellSpursJobQueueAttributeSetGrabParameters(CellSpursJobQueueAttribute *attr, unsigned int maxNumJobsOnSpu, unsigned int maxGrab);

	/* functions to get a parameter for jobDesc pool */
	static inline int cellSpursJobQueueGetJobDescriptorPoolSize(CellSpursJobQueueJobDescriptorPool *pPoolDesc)
	{
		if (pPoolDesc == NULL) {return CELL_SPURS_JOB_ERROR_NULL_POINTER;}
		if (pPoolDesc->nJob64 < 0 || pPoolDesc->nJob128 < 0 ||  pPoolDesc->nJob256 < 0 || pPoolDesc->nJob384 < 0 ||
			pPoolDesc->nJob512 < 0 || pPoolDesc->nJob640 < 0 || pPoolDesc->nJob768 < 0 || pPoolDesc->nJob896 < 0) {
			return CELL_SPURS_JOB_ERROR_INVAL;
		}
		return CELL_SPURS_JOBQUEUE_JOB_DESCRIPTOR_POOL_SIZE(
			pPoolDesc->nJob64, pPoolDesc->nJob128, pPoolDesc->nJob256, pPoolDesc->nJob384,
			pPoolDesc->nJob512, pPoolDesc->nJob640, pPoolDesc->nJob768, pPoolDesc->nJob896);
	}

	/* cellSpursJobQueueCreate internal implementation */
	int _cellSpursCreateJobQueue(
		unsigned int jqRevision,
		unsigned int sdkRevision,
		CellSpurs *pSpurs,
		CellSpursJobQueue *pJobQueue,
		const CellSpursJobQueueAttribute *attr,
		const char *pName,
		uint64_t *pCommandList,
		unsigned int depth,
		unsigned int numSpus,
		const uint8_t priorityTable[CELL_SPURS_MAX_SPU]
		);
	int _cellSpursCreateJobQueueWithJobDescriptorPool(
		unsigned int jqRevision,
		unsigned int sdkRevision,
		CellSpurs *pSpurs,
		CellSpursJobQueue *pJobQueue,
		const CellSpursJobQueueAttribute *attr,
		void *pPool,
		const CellSpursJobQueueJobDescriptorPool *pPoolDesc,
		const char *pName,
		uint64_t *pCommandList,
		unsigned int depth,
		unsigned int numSpus,
		const uint8_t priorityTable[CELL_SPURS_MAX_SPU]
		);

	/* cellSpursJobQueueCreate external interface */
	static inline int cellSpursCreateJobQueue(
		CellSpurs *pSpurs,
		CellSpursJobQueue *pJobQueue,
		const CellSpursJobQueueAttribute *attr,
		const char *pName,
		uint64_t *pCommandList,
		unsigned int depth,
		unsigned int numSpus,
		const uint8_t priorityTable[CELL_SPURS_MAX_SPU]
		)
	{
		int ret = _cellSpursCreateJobQueue(
			CELL_SPURS_JOBQUEUE_REVISION,
			_CELL_SPURS_JQ_INTERNAL_VERSION,
			pSpurs,
			pJobQueue,
			attr,
			pName,
			pCommandList,
			depth,
			numSpus,
			priorityTable
			);
		return ret;
	}
	static inline int cellSpursCreateJobQueueWithJobDescriptorPool(
		CellSpurs *pSpurs,
		CellSpursJobQueue *pJobQueue,
		const CellSpursJobQueueAttribute *attr,
		void *pPool,
		const CellSpursJobQueueJobDescriptorPool *pPoolDesc,
		const char *pName,
		uint64_t *pCommandList,
		unsigned int depth,
		unsigned int numSpus,
		const uint8_t priorityTable[CELL_SPURS_MAX_SPU]
		)
	{
		int ret = _cellSpursCreateJobQueueWithJobDescriptorPool(
			CELL_SPURS_JOBQUEUE_REVISION,
			_CELL_SPURS_JQ_INTERNAL_VERSION,
			pSpurs,
			pJobQueue,
			attr,
			pPool,
			pPoolDesc,
			pName,
			pCommandList,
			depth,
			numSpus,
			priorityTable
			);
		return ret;
	}

	/* Information */
	CellSpurs *cellSpursJobQueueGetSpurs(const CellSpursJobQueue *pJobQueue);
	int cellSpursJobQueueGetHandleCount(const CellSpursJobQueue *pJobQueue);
	int	cellSpursJobQueueGetError(const CellSpursJobQueue *pJobQueue,
				int* exitCode, void** cause);
	int cellSpursJobQueueGetMaxSizeJobDescriptor(const CellSpursJobQueue *pJobQueue);
	int cellSpursGetJobQueueId(const CellSpursJobQueue *pJobQueue, CellSpursWorkloadId *pId);
	int cellSpursJobQueueGetSuspendedJobSize(const CellSpursJobHeader *pJob, size_t sizeJobDesc, enum CellSpursJobQueueSuspendedJobAttribute attr, unsigned int *pSize);

	/* Exception Handling */
	int cellSpursJobQueueSetExceptionEventHandler(
			CellSpursJobQueue* pJobQueue,
			CellSpursJobQueueExceptionEventHandler handler,
			void* arg
		);
	int cellSpursJobQueueSetExceptionEventHandler2(
			CellSpursJobQueue* pJobQueue,
			CellSpursJobQueueExceptionEventHandler2 handler,
			void* arg
		);
	int cellSpursJobQueueUnsetExceptionEventHandler(CellSpursJobQueue* pJobQueue);

	/* State control */
	int cellSpursJobQueueSetWaitingMode(CellSpursJobQueue *pJobQueue, unsigned mode);

	int cellSpursShutdownJobQueue(CellSpursJobQueue *pJobQueue);
	int cellSpursJoinJobQueue(CellSpursJobQueue *pJobQueue, int* exitCode);

	int cellSpursJobQueueOpen(CellSpursJobQueue *pJobQueue, CellSpursJobQueueHandle *handle);
	int cellSpursJobQueueClose(CellSpursJobQueue *pJobQueue, CellSpursJobQueueHandle handle);

	/* Command pushing */
	int _cellSpursJobQueueAllocateJobDescriptorBody(CellSpursJobQueue *eaJobQueue, CellSpursJobQueueHandle handle, 
		size_t sizeJobDesc, unsigned flag, CellSpursJobHeader **eaAllocatedJobDesc);

	int _cellSpursJobQueuePushAndReleaseJobBody(CellSpursJobQueue *eaJobQueue, CellSpursJobQueueHandle handle,
		CellSpursJobHeader *eaJob, size_t sizeJobDesc, unsigned tag, unsigned flag, CellSpursJobQueueSemaphore *eaSemaphore);

	int _cellSpursJobQueuePushJob2Body(CellSpursJobQueue *eaJobQueue, CellSpursJobQueueHandle handle,
		CellSpursJobHeader *eaJob, size_t sizeJobDesc, unsigned tag, unsigned flag, CellSpursJobQueueSemaphore *eaSemaphore);

	int _cellSpursJobQueuePushJobBody2(CellSpursJobQueue *pJobQueue, CellSpursJobQueueHandle handle, CellSpursJobHeader *pJob, size_t sizeJobDesc, unsigned tag, CellSpursJobQueueSemaphore *pSemaphore, bool isExclusive, bool isBlocking);

	int _cellSpursJobQueuePushJobListBody(CellSpursJobQueue *pJobQueue, CellSpursJobQueueHandle handle, CellSpursJobList *pJoblist, unsigned tag, CellSpursJobQueueSemaphore *pSemaphore, bool isBlocking);

	int _cellSpursJobQueuePushFlush(CellSpursJobQueue *pJobQueue, CellSpursJobQueueHandle handle, bool isBlocking);
	int _cellSpursJobQueuePushSync(CellSpursJobQueue *pJobQueue, CellSpursJobQueueHandle handle, unsigned tagMask, bool isBlocking);

	/* backward compatilibity */
	int _cellSpursJobQueuePushBody(CellSpursJobQueue *pJobQueue, CellSpursJobQueueHandle handle, CellSpursJobHeader *pJob, size_t sizeJobDesc, CellSpursJobQueueSemaphore *pSemaphore, bool isBlocking);

	int _cellSpursJobQueuePushJobBody(CellSpursJobQueue *pJobQueue, CellSpursJobQueueHandle handle, CellSpursJobHeader *pJob, size_t sizeJobDesc, unsigned tag, CellSpursJobQueueSemaphore *pSemaphore, bool isBlocking);


	/* external push interfaces */

	static inline int cellSpursJobQueueAllocateJobDescriptor(CellSpursJobQueue *eaJobQueue, CellSpursJobQueueHandle handle, 
		size_t sizeJobDesc, unsigned flag, CellSpursJobHeader **eaAllocatedJobDesc)
	{
		return _cellSpursJobQueueAllocateJobDescriptorBody(eaJobQueue, handle, sizeJobDesc, flag, eaAllocatedJobDesc);
	}

	static inline int cellSpursJobQueuePushAndReleaseJob(CellSpursJobQueue *eaJobQueue, CellSpursJobQueueHandle handle,
		CellSpursJobHeader *eaJob, size_t sizeJobDesc, unsigned tag, unsigned flag, CellSpursJobQueueSemaphore *eaSemaphore)
	{
#ifdef CELL_SPURS_JOBDESCRIPTOR_ERROR_CHECK
		__CELL_SPURS_RETURN_IF(cellSpursJobQueueCheckJob((const CellSpursJob256 *)(uintptr_t)eaJob, sizeJobDesc, cellSpursJobQueueGetMaxSizeJobDescriptor(eaJobQueue)));
#endif
		return _cellSpursJobQueuePushAndReleaseJobBody(eaJobQueue, handle, eaJob, sizeJobDesc, tag, flag, eaSemaphore);
	}

	static inline int cellSpursJobQueuePushJob2(CellSpursJobQueue *eaJobQueue, CellSpursJobQueueHandle handle,
		CellSpursJobHeader *eaJob, size_t sizeJobDesc, unsigned tag, unsigned flag, CellSpursJobQueueSemaphore *eaSemaphore)
	{
#ifdef CELL_SPURS_JOBDESCRIPTOR_ERROR_CHECK
		__CELL_SPURS_RETURN_IF(cellSpursJobQueueCheckJob((const CellSpursJob256 *)(uintptr_t)eaJob, sizeJobDesc, cellSpursJobQueueGetMaxSizeJobDescriptor(eaJobQueue)));
#endif
		return _cellSpursJobQueuePushJob2Body(eaJobQueue, handle, eaJob, sizeJobDesc, tag, flag, eaSemaphore);
	}

	static inline int cellSpursJobQueuePush(CellSpursJobQueue *pJobQueue, CellSpursJobQueueHandle handle, CellSpursJobHeader *pJob, size_t sizeJobDesc, CellSpursJobQueueSemaphore *pSemaphore)
	{
#ifdef CELL_SPURS_JOBDESCRIPTOR_ERROR_CHECK
		__CELL_SPURS_RETURN_IF(cellSpursJobQueueCheckJob((const CellSpursJob256 *)(uintptr_t)pJob, sizeJobDesc, cellSpursJobQueueGetMaxSizeJobDescriptor(pJobQueue)));
#endif
		return _cellSpursJobQueuePushJobBody2(pJobQueue, handle, pJob, sizeJobDesc, 0, pSemaphore, false, true);
	}

	static inline int cellSpursJobQueuePushJob(CellSpursJobQueue *pJobQueue, CellSpursJobQueueHandle handle, CellSpursJobHeader *pJob, size_t sizeJobDesc, unsigned tag, CellSpursJobQueueSemaphore *pSemaphore)
	{
#ifdef CELL_SPURS_JOBDESCRIPTOR_ERROR_CHECK
		__CELL_SPURS_RETURN_IF(cellSpursJobQueueCheckJob((const CellSpursJob256 *)(uintptr_t)pJob, sizeJobDesc, cellSpursJobQueueGetMaxSizeJobDescriptor(pJobQueue)));
#endif
		return _cellSpursJobQueuePushJobBody2(pJobQueue, handle, pJob, sizeJobDesc, tag, pSemaphore, false, true);
	}

	static inline int cellSpursJobQueuePushExclusiveJob(CellSpursJobQueue *pJobQueue, CellSpursJobQueueHandle handle, CellSpursJobHeader *pJob, size_t sizeJobDesc, unsigned tag, CellSpursJobQueueSemaphore *pSemaphore)
	{
#ifdef CELL_SPURS_JOBDESCRIPTOR_ERROR_CHECK
		__CELL_SPURS_RETURN_IF(cellSpursJobQueueCheckJob((const CellSpursJob256 *)(uintptr_t)pJob, sizeJobDesc, cellSpursJobQueueGetMaxSizeJobDescriptor(pJobQueue)));
#endif
		return _cellSpursJobQueuePushJobBody2(pJobQueue, handle, pJob, sizeJobDesc, tag, pSemaphore, true, true);
	}

	static inline int cellSpursJobQueueTryPush(CellSpursJobQueue *pJobQueue, CellSpursJobQueueHandle handle, CellSpursJobHeader *pJob, size_t sizeJobDesc, CellSpursJobQueueSemaphore *pSemaphore)
	{
#ifdef CELL_SPURS_JOBDESCRIPTOR_ERROR_CHECK
		__CELL_SPURS_RETURN_IF(cellSpursJobQueueCheckJob((const CellSpursJob256 *)(uintptr_t)pJob, sizeJobDesc, cellSpursJobQueueGetMaxSizeJobDescriptor(pJobQueue)));
#endif
		return _cellSpursJobQueuePushJobBody2(pJobQueue, handle, pJob, sizeJobDesc, 0, pSemaphore, false, false);
	}

	static inline int cellSpursJobQueueTryPushJob(CellSpursJobQueue *pJobQueue, CellSpursJobQueueHandle handle, CellSpursJobHeader *pJob, size_t sizeJobDesc, unsigned tag, CellSpursJobQueueSemaphore *pSemaphore)
	{
#ifdef CELL_SPURS_JOBDESCRIPTOR_ERROR_CHECK
		__CELL_SPURS_RETURN_IF(cellSpursJobQueueCheckJob((const CellSpursJob256 *)(uintptr_t)pJob, sizeJobDesc, cellSpursJobQueueGetMaxSizeJobDescriptor(pJobQueue)));
#endif
		return _cellSpursJobQueuePushJobBody2(pJobQueue, handle, pJob, sizeJobDesc, tag, pSemaphore, false, false);
	}

	static inline int cellSpursJobQueueTryPushExclusiveJob(CellSpursJobQueue *pJobQueue, CellSpursJobQueueHandle handle, CellSpursJobHeader *pJob, size_t sizeJobDesc, unsigned tag, CellSpursJobQueueSemaphore *pSemaphore)
	{
#ifdef CELL_SPURS_JOBDESCRIPTOR_ERROR_CHECK
		__CELL_SPURS_RETURN_IF(cellSpursJobQueueCheckJob((const CellSpursJob256 *)(uintptr_t)pJob, sizeJobDesc, cellSpursJobQueueGetMaxSizeJobDescriptor(pJobQueue)));
#endif
		return _cellSpursJobQueuePushJobBody2(pJobQueue, handle, pJob, sizeJobDesc, tag, pSemaphore, true, false);
	}

	static inline int cellSpursJobQueuePushJobList(CellSpursJobQueue *pJobQueue, CellSpursJobQueueHandle handle, CellSpursJobList *pJoblist, unsigned tag, CellSpursJobQueueSemaphore *pSemaphore)
	{
#ifdef CELL_SPURS_JOBDESCRIPTOR_ERROR_CHECK
		for(unsigned index = 0; index < pJoblist->numJobs; index++) {
			CellSpursJob256*	pJob = (CellSpursJob256*)(uintptr_t)pJoblist->eaJobList + pJoblist->sizeOfJob * index;
			__CELL_SPURS_RETURN_IF(cellSpursJobQueueCheckJob((const CellSpursJob256 *)pJob, pJoblist->sizeOfJob, cellSpursJobQueueGetMaxSizeJobDescriptor(pJobQueue)));
		}
#endif
		return _cellSpursJobQueuePushJobListBody(pJobQueue, handle, pJoblist, tag, pSemaphore, true);
	}

	static inline int cellSpursJobQueueTryPushJobList(CellSpursJobQueue *pJobQueue, CellSpursJobQueueHandle handle, CellSpursJobList *pJoblist, unsigned tag, CellSpursJobQueueSemaphore *pSemaphore)
	{
#ifdef CELL_SPURS_JOBDESCRIPTOR_ERROR_CHECK
		for(unsigned index = 0; index < pJoblist->numJobs; index++) {
			CellSpursJob256*	pJob = (CellSpursJob256*)(uintptr_t)pJoblist->eaJobList + pJoblist->sizeOfJob * index;
			__CELL_SPURS_RETURN_IF(cellSpursJobQueueCheckJob((const CellSpursJob256 *)pJob, pJoblist->sizeOfJob, cellSpursJobQueueGetMaxSizeJobDescriptor(pJobQueue)));
		}
#endif
		return _cellSpursJobQueuePushJobListBody(pJobQueue, handle, pJoblist, tag, pSemaphore, false);
	}

	static inline int cellSpursJobQueuePushFlush(CellSpursJobQueue *pJobQueue, CellSpursJobQueueHandle handle)
	{
		return _cellSpursJobQueuePushFlush(pJobQueue, handle, true);
	}

	static inline int cellSpursJobQueueTryPushFlush(CellSpursJobQueue *pJobQueue, CellSpursJobQueueHandle handle)
	{
		return _cellSpursJobQueuePushFlush(pJobQueue, handle, false);
	}

	static inline int cellSpursJobQueuePushSync(CellSpursJobQueue *pJobQueue, CellSpursJobQueueHandle handle, unsigned tagMask)
	{
		return _cellSpursJobQueuePushSync(pJobQueue, handle, tagMask, true);
	}

	static inline int cellSpursJobQueueTryPushSync(CellSpursJobQueue *pJobQueue, CellSpursJobQueueHandle handle, unsigned tag)
	{
		return _cellSpursJobQueuePushSync(pJobQueue, handle, tag, false);
	}

	int cellSpursJobQueueSendSignal(CellSpursJobQueueWaitingJob *job);

#ifdef __cplusplus
}
#endif

#include <cell/spurs/job_queue_cpp_types.h>

#endif /* __CELL_SPURS_JOBQUEUE_H__ */
