/* SCE CONFIDENTIAL
 * PlayStation(R)3 Programmer Tool Runtime Library 475.001
 * Copyright (C) 2012 Sony Computer Entertainment Inc.
 * All Rights Reserved.
 */

#ifndef __CELL_SAIL_COMPOSER_H__
#define __CELL_SAIL_COMPOSER_H__


#if defined(_LANGUAGE_C_PLUS_PLUS)||defined(__cplusplus)||defined(c_plusplus)
extern "C" {
#endif




typedef struct CellSailComposerEsAudioParameter {
    int codecType;
    int flags;       // must be 0
    int reserved0;
    int reserved1;
    int maxBitrate;
    int aveBitrate;
    int sampleScale;
    int reserved2[3];
    CellSailAudioFormat format;
    void *pSpecificInfo;
} CellSailComposerEsAudioParameter;


typedef struct CellSailComposerEsVideoParameter {
    int codecType;
    int flags;       // must be 0
    int majorMode;
    int minorMode;
    int maxBitrate;
    int aveBitrate;
    int sampleScale;
    int gopDuration;
    int reserved[2];
    CellSailVideoFormat format;
    void *pSpecificInfo;
} CellSailComposerEsVideoParameter;



/*
 * Audio AU info for user defined muxer
 * see cellSailComposerGetAudioAu(), cellSailComposerTryGetAudioAu()
 */
typedef struct CellSailComposerAudioAuInfo {
    uint8_t *pAu;        /// pointer to AU(AccessUnit)
    size_t   auSize;       /// size of AU
    int64_t  pts;          /// PTS(presentation time stamp)
    uint64_t reserved;     /// not available for now
    uint32_t sessionId;    /// not available for now
    void    *specificInfo; /// not available for now
} CellSailComposerAudioAuInfo;


/*
 * Video AU info for user defined muxer
 * see cellSailComposerGetVideoAu(), cellSailComposerTryGetVideoAu()
 */
typedef struct CellSailComposerVideoAuInfo {
    uint8_t *pAu;   // pointer to AU(AccessUnit)
    size_t   auSize;  // pointer to AU(AccessUnit)
    int64_t  pts[3]; // PTS(presentation time stamp) only pts[0] is available
    int64_t  dts;         // DTS(decode time stamp)
    uint64_t reserved;    // not available for now
    uint32_t sessionId;   // not available for now
    void    *specificInfo;// not available for now
} CellSailComposerVideoAuInfo;


typedef struct CellSailComposer {
    uint64_t internalData[40];
} CellSailComposer;

/*
 * to init user defined muxer by libsail_rec thread
 */
typedef int (*CellSailComposerFuncMakeup)(void *pArg);

/*
 * to finalize user defined muxer by libsail_rec thread
 */
typedef int (*CellSailComposerFuncCleanup)(void *pArg);

/*
 * to prepare for stream open.
 * call cellSailComposerNotifyCallCompleted() after operation is completed.
 *
 * @param pBaseUri  string passed to cellSailRecorderOpenStream()
 */
typedef void (*CellSailComposerFuncOpenStream)(
    void                                *pArg,
    void                                *pParam,
    const char                          *pBaseUri);

/*
 * to close stream.
 * call cellSailComposerNotifyCallCompleted() after operation is completed.
 */
typedef void (*CellSailComposerFuncCloseStream)(void *pArg);

/*
 * to start recording session.
 * call cellSailComposerNotifyCallCompleted() after operation is completed.
 *
 * @param pPath      string passed to cellSailRecorderStart()
 * @param sessionId  not available for now
 */
typedef void (*CellSailComposerFuncStart)(void       *pArg,
                                          const char *pPath,
                                          uint32_t    sessionId);

/*
 * to stop recording session.
 * this callback is called after all audio & video au are retrieved by user.
 * call cellSailComposerNotifyCallCompleted() after operation is completed.
 *
 * @param error      !=0 if stop has happend due to error
 */
typedef void (*CellSailComposerFuncStop)(void *pArg, int error);

/*
 * (not suppoeted yet)
 */
typedef void (*CellSailComposerFuncCancel)(void *pArg);


typedef void (*CellSailComposerFuncEsAudioEnabled)(
    void                                   *pArg,
    int                                     audioEsIndex,
    CellSailComposerEsAudioParameter const *pParam);

typedef void (*CellSailComposerFuncEsVideoEnabled)(
    void                                   *pArg,
    int                                     videoEsIndex,
    CellSailComposerEsVideoParameter const *pParam);


/*
 * to tell muxer that audio au is ready to get.
 *
 * @audioEsIndex  must be ==0 so far
 * @num           number of generated au
 */
typedef int (*CellSailComposerFuncEsAudioAuOut)(
    void                           *pArg,
    int                             audioEsIndex,
    int                             num);

/*
 * to tell muxer that video au is ready to get.
 *
 * @videoEsIndex  must be ==0 so far
 * @num           number of generated au
 */
typedef int (*CellSailComposerFuncEsVideoAuOut)(
    void                           *pArg,
    int                             videoEsIndex,
    int                             num);

/*
 * to tell muxer that audio encoding has finished.
 * no more au to be generated after this callback.
 * i.e. CellSailComposerFuncAudioAuOut never called after this callback within a session.
 */
typedef int (*CellSailComposerFuncEsAudioEos)(
    void                           *pArg,
    int                             audioEsIndex);

/*
 * to tell muxer that video encoding has finished.
 * no more au to be generated after this callback.
 * i.e. CellSailComposerFuncVideoAuOut never called after this callback within a session.
 */
typedef int (*CellSailComposerFuncEsVideoEos)(
    void                           *pArg,
    int                             videoEsIndex);

/*
 * see cellSailComposerInitialize()
 */
typedef struct CellSailComposerFuncs {
    CellSailComposerFuncMakeup   pMakeup;
    CellSailComposerFuncCleanup  pCleanup;

    CellSailComposerFuncOpenStream  pOpenStream;
    CellSailComposerFuncCloseStream pCloseStream;
    CellSailComposerFuncStart       pStart;
    CellSailComposerFuncStop        pStop;
    CellSailComposerFuncCancel      pCancel;

    CellSailComposerFuncEsAudioEnabled pEsAudioEnabled;
    CellSailComposerFuncEsVideoEnabled pEsVideoEnabled;
    CellSailComposerFuncEsAudioAuOut   pEsAudioAuOut;
    CellSailComposerFuncEsVideoAuOut   pEsVideoAuOut;
    CellSailComposerFuncEsAudioEos     pEsAudioEos;
    CellSailComposerFuncEsVideoEos     pEsVideoEos;

    void* reserved[4];
} CellSailComposerFuncs;


/*
 * init user defined muxer
 * @param pFunc  callback functions
 * @param pArg   callback arg
 */
int cellSailComposerInitialize(CellSailComposer            *pSelf,
                               const CellSailComposerFuncs *pFuncs,
                               void                        *pArg);

/*
 * finalize user defined muxer
 */
int cellSailComposerFinalize(CellSailComposer *pSelf);



int cellSailComposerGetEsAudioParameter(
    CellSailComposer                 *pSelf,
    int                               esAudioIndex,
    CellSailComposerEsAudioParameter *pAudioParam,
    size_t                            paramSize);
int cellSailComposerGetEsVideoParameter(
    CellSailComposer                 *pSelf,
    int                               esVideoIndex,
    CellSailComposerEsVideoParameter *pVideoParam,
    size_t                            paramSize);


/*
 * get generated audio(s) au (blocking call)
 * you can get the same auInfo again unless call cellSailComposerReleaseAudioAu()
 *
 * @param [out]  pInfo  array of AuInfo
 * @param [in]   num    length of pInfo
 * @param [out]  pRemaining  remaining number of AuInfo that was not stored to pInfo
 * @param [in]   timeout     timeout param to be passed to sys_lwcond_wait()
 * @ret  number of AuInfo you actually got
 */
int cellSailComposerGetEsAudioAu(CellSailComposer            *pMuxer,
                                 int                          audioEsIndex,
                                 CellSailComposerAudioAuInfo *pInfo,
                                 int                          num,
                                 int                         *pRemaining,
                                 uint64_t                     timeout);

/*
 * get generated video au (blocking call)
 */
int cellSailComposerGetEsVideoAu(CellSailComposer            *pMuxer,
                                 int                          videoEsIndex,
                                 CellSailComposerVideoAuInfo *pInfo,
                                 int                          num,
                                 int                         *pRemaining,
                                 uint64_t                     timeout);

/*
 * get generated audio(s) au (non-blocking call)
 */
int cellSailComposerTryGetEsAudioAu(CellSailComposer            *pMuxer,
                                    int                          audioEsIndex,
                                    CellSailComposerAudioAuInfo *pInfo,
                                    int                          num,
                                    int                          *pRemaining);

/*
 * get generated video(s) au (non-blocking call)
 */
int cellSailComposerTryGetEsVideoAu(CellSailComposer            *pMuxer,
                                    int                          videoEsIndex,
                                    CellSailComposerVideoAuInfo *pInfo,
                                    int                          num,
                                    int                         *pRemaining);

/*
 * release audio AuInfo
 * @param num  number of AuInfo you want to release
 * @ret  number of AuInfo you actually released
 */
int cellSailComposerReleaseEsAudioAu(CellSailComposer *pMuxer,
                                     int               audioEsIndex,
                                     int               num);

/*
 * release video AuInfo
 */
int cellSailComposerReleaseEsVideoAu(CellSailComposer *pMuxer,
                                     int               videoEsIndex,
                                     int               num);

/*
 * call if asynchronous operation has finished
 * @param status  result of asynchronous operation
 *
 * see
 * - CellSailComposerFuncOpenStream
 * - CellSailComposerFuncCloseStream
 * - CellSailComposerFuncStart
 * - CellSailComposerFuncStop
 */
void cellSailComposerNotifyCallCompleted(CellSailComposer *pSelf,
                                         int               status);

/*
 * call if unexpected error has happened during recording session.
 * @isFatal  set !=0 to stop recording
 */
void cellSailComposerNotifySessionError(CellSailComposer *pMuxer,
                                        int               error,
                                        int               isFatal);


#if defined(_LANGUAGE_C_PLUS_PLUS)||defined(__cplusplus)||defined(c_plusplus)
}
#endif

#endif /* __CELL_SAIL_COMPOSER_H__ */
