/* SCE CONFIDENTIAL
PlayStation(R)3 Programmer Tool Runtime Library 475.001 
* Copyright (C) 2010 Sony Computer Entertainment Inc.
* All Rights Reserved.
*/


#ifndef _MULTISTREAM_MP3_SPURS_LIBRARY_H_
#define _MULTISTREAM_MP3_SPURS_LIBRARY_H_


#include <cell/spurs/types.h>


#if defined(_LANGUAGE_C_PLUS_PLUS)||defined(__cplusplus)||defined(c_plusplus)
extern "C" {
#endif

// See CellMSMP3DecoderInitParams->flags
#define CELL_MSMP3_DISABLE_SPU_PRINTF_SERVER  (8)

// Size of temp buffer required by cellMSMP3DecoderInit(), once initialized can be freed
#define CELL_MSMP3_TEMP_INIT_BUFFER_SIZE      (1024 * 96)	

typedef struct CellMSMP3DecoderPacketInfo
{
    unsigned int Sync;
	unsigned int ID;
	unsigned int Layer;
	unsigned int ProtBit;
	unsigned int BitRate;
	unsigned int Frequency;
	unsigned int PadBit;
	unsigned int PrivBit;
	unsigned int Mode;
	unsigned int ModeExt;
	unsigned int Copy;
	unsigned int Home;
	unsigned int Emphasis;
    unsigned int OutputSizeBytes;
	unsigned int PacketSizeBytes;
	unsigned int ID3;			// 16 bit value (VV,vv), VV=Major version, vv=revision (example: ID3v2.4). Set to 0 if packet is not an ID3 packet
	unsigned int Tag;			// 0 = packet is not a Tag packet, 1=Packet is a Tag packet

    
} CellMSMP3DecoderPacketInfo;



typedef enum CELL_MSMP3_DECODER_RESULT
{
    CELL_MSMP3_RESULT_OK = 0,
    CELL_MSMP3_RESULT_ERR_TOO_FEW_CONTEXTS,
    CELL_MSMP3_RESULT_ERR_INTERNAL_DECODING,
    CELL_MSMP3_RESULT_ERR_UNHANDLED_CHANNELS,
    CELL_MSMP3_RESULT_ERR_CHANNEL_MISMATCH,
    CELL_MSMP3_RESULT_ERR_INVALID_DATA,
} CELL_MSMP3_DECODER_RESULT;



typedef struct CellMSMP3Context
{
	float decodeBuff[544];
	float oldBlock[576];
	char  oldFrame[512];
} CellMSMP3Context;

typedef struct CellMSMP3DecoderStreamHead
{
    int                  nChannels;				// number of channels in the stream
    void*                pMp3DataBuffer;		// pointer to the MP3 data
    int                  nMp3BufferSizeBytes;	// number of bytes in the MP3 buffer
    int                  nMp3BufferBytesRead;	// number of bytes read from the MP3 data buffer

    short*               pPcmBuffer;			// pointer to the output PCM data buffer
    int                  nPcmBufferSizeBytes;	// size in bytes of the PCM output buffer
    int                  nPcmBufferBytesWritten;// number of bytes written into the output PCM buffer
    int                  internalCount;			// internal counter, initialized to 1

    short*               pPcmBuffer2;           // if NULL, stereo will be interleaved otherwise the size and bytes written will match pPcmBuffer
    CELL_MSMP3_DECODER_RESULT eReturnCode;		// return code from decode processing
    int                  nBlocksDecoded;		// count on number of decoded blocks
    int                  nSkipFirstPacket;		// flag to skip first packet (1) used for looping
} CellMSMP3DecoderStreamHead;


typedef struct CellMSMP3DecoderStreamInstance
{
    CellMSMP3DecoderStreamHead  head;
    CellMSMP3Context*           pContexts[2]; // one per output channel
    int                         pad[2];
} CellMSMP3DecoderStreamInstance;



typedef struct CellMSMP3DecoderInitParams
{
    CellSpurs*  pSpurs;         // pointer to SPURs instance
    uint8_t*    pPriorities;    // pointer to array of 8 uint8 priorities for the SPUs
    int         flags;          // flags for MP3 decoder
    int         nMaxStreams;    // maximum streams to update per call to cellMSMP3DecoderProcess
} CellMSMP3DecoderInitParams;



//////////////////////////////////////////////////////////////////////////////////
// cellMSMP3DecoderGetPacketInfo                                                 //
//////////////////////////////////////////////////////////////////////////////////
// Returns information of an MP3 data packet
// 
// Requires:
//   void* pMp3Data					Pointer to an MP3 data packet
//   CellMSMP3DecoderPacketInfo		Pointer to a CellMSMP3DecoderPacketInfo structure (information will be stored here)
// 
// Returns:
//   0 = OK
//  -1 = Failed
//////////////////////////////////////////////////////////////////////////////////
int cellMSMP3DecoderGetPacketInfo( const void* pMp3Data, CellMSMP3DecoderPacketInfo* pPacket );


//////////////////////////////////////////////////////////////////////////////////
// cellMSMP3DecoderLibGetRequiredSize                                            //
//////////////////////////////////////////////////////////////////////////////////
// Calculates the required size of RAM needed, based on the maximum number of MP3 stream instances.
// 
// Requires:
//    CellMSMP3DecoderInitParams * pParams	Pointer to a cellMSMP3DecoderInitParams structure
//    int * pnReturnedSize					Pointer to an int, which will contain the size (in bytes) of a buffer for initialization
// 
// pParams required information:
//   nMaxStreams must be filled in by the user before this function is called to a range of 0-n
// 
// Returns:
//   0 = OK (size is returned via the *pnReturnedSize pointer)
//  -1 = Failed
//////////////////////////////////////////////////////////////////////////////////
int cellMSMP3DecoderLibGetRequiredSize( const CellMSMP3DecoderInitParams* pParams, int* pnRequiredSize );


//////////////////////////////////////////////////////////////////////////////////
// cellMSMP3DecoderInit                                                          //
//////////////////////////////////////////////////////////////////////////////////
// Initializes the MP3 Decoder.
// Requires the user to allocation RAM prior to calling this function. The buffer size is calculated by calling cellMSMP3DecoderLibGetRequiredSize.
// 
// Requires:
//   void * pDecoderMem						Pointer to a decoder work buffer (user must allocate this). Must be 128 byte alligned
//   int nDecoderMemSize					Size of decoder work buffer (calculated by cellMSMP3DecoderLibGetRequiredSize)
//   void * pTempMemory						Pointer to temporary buffer used for initialization
//   int	nTempMemSize					Size of temporary buffer used for intialization only, must be at least CELL_MSMP3_TEMP_INIT_BUFFER_SIZE size.
//   CellMSMP3DecoderInitParams *pParams	Pointer to a cellMSMP3DecoderInitParams structure
// 
// Returns:
//   0 = OK
//  <0 = Failed
//  -1 = NULL pointers passed via pDecoderMem, pTempMemory or pParams
//  -2 = pDecoderMem or pTempMemory is not 128 byte alligned
//  -3 = pParams information is invalid
//  -4 = Invalid memory sizes
//  -5 = Spurs task creation failed
//////////////////////////////////////////////////////////////////////////////////
int cellMSMP3DecoderInit( void* pDecoderMem, const int nDecoderMemSize, void* pTempMemory, int nTempMemSize, CellMSMP3DecoderInitParams* pParams );


//////////////////////////////////////////////////////////////////////////////////
// cellMSMP3DecoderProcess                                                       //
//////////////////////////////////////////////////////////////////////////////////
// Processes any MP3 stream that needs to create more PCM output data.
// To include a stream to be decoded, the celMP3DecoderAddStruct function must be called for each stream prior to calling cellMSMP3DecoderProcess
// The list of streams to be decoded is cleared after each call to cellMSMP3DecoderProcess
// 
// Requires:
//   voice * pDecoderMem		Pointer to a decoder work buffer
// 
// Returns:
//   0 = OK
//  -1= Failed
// 
// Notes:
//  This is a blocking function. Blocking will occur until SPU processing of MP3 data is complete.
//////////////////////////////////////////////////////////////////////////////////
int cellMSMP3DecoderProcess( void* pDecoderMem );


//////////////////////////////////////////////////////////////////////////////////
// cellMSMP3DecoderShutdown                                                      //
//////////////////////////////////////////////////////////////////////////////////
// Shuts down the MP3Decoder. Once this function returns, it is safe to free any allocated memory.
// 
// Requires:
//   voice * pDecoderMem			Pointer to a decoder work buffer
// 
// Returns:
//   0 = OK
//  -1 = Failed
//////////////////////////////////////////////////////////////////////////////////
int cellMSMP3DecoderShutdown( void* pDecoderMem );



//////////////////////////////////////////////////////////////////////////////////
// cellMSMP3DecoderInitStruct                                                    //
//////////////////////////////////////////////////////////////////////////////////
// Initializes an instance of a decoder stream structure.
// The user is responsible for handling and memory management of these structures
// 
// Requires:
//   CellMSMP3DecoderStreamInstance* pInstance			Pointer to a CellMSMP3DecoderStreamInstance structure
// 
// Returns:
//   0 = OK
//  <0 = Failed
//  -1 = User has passed NULL pointer
//  -2 = Too many channels
// 
// Notes:
//   pInstance must be 16 byte aligned
//////////////////////////////////////////////////////////////////////////////////
int cellMSMP3DecoderInitStruct( CellMSMP3DecoderStreamInstance* pInstance ); 


//////////////////////////////////////////////////////////////////////////////////
// cellMSMP3DecoderAddStruct                                                     //
//////////////////////////////////////////////////////////////////////////////////
// Adds a decoder stream instance to the process list.
// 
// Requires:
//   voice * pDecoderMem								Pointer to a decoder work buffer
//   CellMSMP3DecoderStreamInstance* pInstance			Pointer to a CellMSMP3DecoderStreamInstance structure
// 
// Returns:
//   0 = OK
//  -1 = Failed
// 
// Notes:
//   List is processed by calling cellMSMP3DecoderProcess.
//   The list is cleared after cellMSMP3DecoderProcess has been called
//   The user is responsible for setting up the pointers to the input and output data buffers and sizes
//////////////////////////////////////////////////////////////////////////////////
int cellMSMP3DecoderAddStruct( void* pDecoderMem, const CellMSMP3DecoderStreamInstance* pInstance );  



#if defined(_LANGUAGE_C_PLUS_PLUS)||defined(__cplusplus)||defined(c_plusplus)
}
#endif

#endif //_MULTISTREAM_MP3_SPURS_LIBRARY_H_