/* SCE CONFIDENTIAL
PlayStation(R)3 Programmer Tool Runtime Library 475.001
* Copyright (C) 2010 Sony Computer Entertainment Inc.
* All Rights Reserved.
*/


//DOM-IGNORE-BEGIN
#ifndef _CELL_LIBMD5_SPU_H_
#define _CELL_LIBMD5_SPU_H_


#if defined(_LANGUAGE_C_PLUS_PLUS)||defined(__cplusplus)||defined(c_plusplus)
extern "C" {
#endif	/* defined(_LANGUAGE_C_PLUS_PLUS)||defined(__cplusplus)||defined(c_plusplus) */


#define CELL_MD5_DIGEST_SIZE    16

#include "libhash.h"

#define CELL_MD5_BUFFQSIZE 32
	//DOM-IGNORE-END

	typedef struct {
		CellSpursTaskId task_id;
		CellSpursTaskset2 *taskset;
		CellSpursQueue *queue;
		CellSpursEventFlag *event;
		void *context_store;
		void *qdata;
		int free_flags;
		void *_internal_use_;
	} CellMd5SpuInstance;//*** Note to editor, the contents of this structure should be hidden, in the same way that CellSpurs is ***
	//Summary:
	//Instance of an MD5 SPU task
	//
	//Description:
	//This is the datatype of a MD5 SPU instance, and holds information about the SPURS usage for this particular instance.
	//It is not necessary or appropriate for the application to handle the members of this datatype directly.
	typedef struct  {
		CellMd5SpuInstance* md5_instance;
		uint32_t task_id;
		uint32_t context_id;
		int		 task_idle;
	}cellMd5DataStream;
	
	typedef struct{
		cellMd5DataStream* data_streams;	
		uint64_t num_streams;

	}CellMd5SpuStreamsInstance;//*** Note to editor, the contents of this structure should be hidden, in the same way that CellSpurs is ***
	//Summary:
	//Instance of MD5 SPU work area for handling concurrent data stream
	//
	//Description:
	//This is the datatype of a MD5 SPU instance, and holds information about each stream instance and SPURS usage for a particular instance.
	//It is not necessary or appropriate for the application to handle the members of this datatype directly.
	




	/* Summary
	Create a MD5 SPURS task.[deprecated function, maintained for compatibility]
	Conditions
	Multi-thread safe.
	Parameters
	md5 :				Pointer to a MD5 SPURS work area.
	taskset :			Pointer to a SPURS task set instance.
	queue :				Address of a SPURS event queue instance that can be
						used by this SPURS task, or NULL if this should be
						allocated dynamically.
	event :				Address of a SPURS event flag instance that can be used
						by this SPURS task, or NULL if this should be allocated
						dynamically.
	context_store :		Address of the context storage area (16\-byte aligned,
						preferably 128\-byte aligned), or NULL if this should
						be allocated dynamically.
	queue_depth :		Number of entries in the queue. This should be at least
						1+(number of buffers).
	Returns
	<table>
	Macro                                   Value        \Description
	-------------------------------------   -----------  --------------------------
	CELL_OK                                 0x00000000   Function returned successfully
	CELL_ERROR_HASH_NULL_POINTER            0x80f00001   <c>taskset</c> or <c>md5</c>
														 is a null pointer.
	CELL_ERROR_HASH_QUEUE_CREATE_FAILED     0x80f00002   SPURS event queue could not be created, 
														 or the event queue could not be 
														 attached to a SPURS queue.
	CELL_ERROR_HASH_FLAG_CREATE_FAILED      0x80f00003   SPURS event flag could not be created, 
														 or the event flag could not be attached
														 to a SPURS queue.
	CELL_ERROR_HASH_INVALID_ALIGN           0x80f00004   <c>queue</c> or <c>event</c> is not 
														 aligned to a 128\-byte boundary, or 
														 <c>context_store</c> is not aligned to 
														 a 16\-byte boundary.
	CELL_ERROR_HASH_INVALID_NOMEM           0x80f00005   Insufficient memory.
	CELL_ERROR_HASH_TASK_CREATE_FAILED      0x80f00006   Failed to create SPURS task.

	</table>


	Description
	This function is maintained for purposes of compatibility with earlier versions of Hash Function libraries. 
	From now on, use the<c>cellMd5SpuCreateTask2</c>.

	This function creates a new SPURS task for calculating the
	MD5 hash value of some data. For <c>md5</c> and <c>taskset</c>,
	these should be provided to the function and allocated and
	initialized accordingly.

	The structure <c>md5</c> is used to hold information about
	the MD5 SPURS task that is created by this function and
	should not be altered.

	For <c>queue</c>, <c>event</c>, <c>context_store</c>, these
	can either be provided to the function, or set to NULL. When
	one or more of these parameters is set to NULL, the memory
	required by the parameter(s) would be created
	during the initialization process. Internally, <c>memalign</c>
	is used to allocate memory, so if you want to use your own
	memory allocation functions, then memory should be allocated
	to these parameters before calling this function.

	For efficiency, ensure that the queue depth is at least the
	same size as the number of buffers that you will send as well
	as one extra buffer for ending the task.
	See Also
	<c>cellMd5SpuEndTask</c>                                                                              */
	int cellMd5SpuCreateTask(CellMd5SpuInstance *md5,
		CellSpursTaskset *taskset,
		CellSpursQueue *queue, 
		CellSpursEventFlag *event, 
		void *context_store,
		int queue_depth);


/* Summary
	Create a Md5 SPURS task.
	Conditions
	Multi-thread safe.
	Parameters
	md5 :					Pointer to a MD5 SPURS work area.
	taskset :				Pointer to a SPURS task set instance.
	queue :					Address of a SPURS event queue instance that can be
							used by this SPURS task, or NULL if this should be
							allocated dynamically.
	event :					Address of a SPURS event flag instance that can be used
							by this SPURS task, or NULL if this should be allocated
							dynamically.
	context_store :			Address of the context storage area (16\-byte aligned,
							preferably 128\-byte aligned), or NULL if this should
							be allocated dynamically.
	queue_depth :			Number of entries in the queue. This should be at least
							1+(number of buffers).
	spu_dma_buffer_size :	Specify the DMA buffer size used by this SPURS task, or specify NULL 
							if default buffer size of 2K is to be used 
	Returns
	<table>
	Macro                                   Value        \Description
	-------------------------------------   -----------  --------------------------
	CELL_OK                                 0x00000000   Function returned successfully
	CELL_ERROR_HASH_NULL_POINTER            0x80f00001   <c>taskset</c> or <c>md5</c>
														 is a null pointer.
	CELL_ERROR_HASH_QUEUE_CREATE_FAILED     0x80f00002   SPURS event queue could not be created, 
														 or the event queue could not be 
														 attached to a SPURS queue.
	CELL_ERROR_HASH_FLAG_CREATE_FAILED      0x80f00003   SPURS event flag could not be created, 
														 or the event flag could not be attached
														 to a SPURS queue.
	CELL_ERROR_HASH_INVALID_ALIGN           0x80f00004   <c>queue</c> or <c>event</c> is not 
														 aligned to a 128\-byte boundary, or 
														 <c>context_store</c> is not aligned to 
														 a 16\-byte boundary.
	CELL_ERROR_HASH_INVALID_NOMEM           0x80f00005   Insufficient memory.
	CELL_ERROR_HASH_TASK_CREATE_FAILED      0x80f00006   Failed to create SPURS task.
	CELL_ERROR_HASH_INVALID_BUFFER_SIZE		0x80f00008   <c>spu_dma_buffer_size</c> is not a multiple of 16 
														 or the size is greater than 16K 
	

	</table>


	Description:
	This function is maintained for purposes of compatibility with earlier versions of libhash. 
	From now on, use the<c>cellMd5SpuCreateTask2</c>.
	This function creates a new SPURS task for calculating the
	MD5 hash value of some data. For <c>md5</c> and <c>taskset</c>,
	these should be provided to the function and allocated and
	initialized accordingly.

	The structure <c>md5</c> is used to hold information about
	the md5 SPURS task that is created by this function and
	should not be altered.

	For <c>queue</c>, <c>event</c>, <c>context_store</c>, these
	can either be provided to the function, or set to NULL. When
	one or more of these parameters is set to NULL, the memory
	required by the parameter(s) would be created
	during the initialization process. Internally, <c>memalign</c>
	is used to allocate memory, so if you want to use your own
	memory allocation functions, then memory should be allocated
	to these parameters before calling this function.

	For efficiency, ensure that the queue depth is at least the
	same size as the number of buffers that you will send as well
	as one extra buffer for ending the task.
	See Also
	<c>cellMd5SpuEndTask2</c>                                                                              */
	int cellMd5SpuCreateTask2(CellMd5SpuInstance *md5,
		CellSpursTaskset2 *taskset,
		CellSpursQueue *queue, 
		CellSpursEventFlag *event, 
		int queue_depth,
		int spu_dma_buffer_size);

	/* Summary
	Process a chunk of data.
	Conditions
	Multi-thread safe.
	Parameters
	md5 :  Pointer to a MD5 SPURS work area
	data :    Effective address of data to be processed on the PPU
	(16\-byte aligned)
	size :    Number of bytes to process
	flag :    Event flag which is set once processing is complete
	wait :    Waiting mode
	Returns
	<c>CELL_OK</c>(0) is returned for success.

	If <c>md5==NULL</c> or <c>data==0</c>, then 
	<c>CELL_ERROR_HASH_NULL_POINTER</c> is returned.

	If any other error occurs, an error code from <c>cellSpursQueuePush</c>
	(if <c>wait==1</c>) or cell <c>SpursQueueTryPush</c> (if <c>wait==0</c>)
	is returned.
	Description
	This function adds more data to the MD5 SPURS task for
	processing. If the task is already processing data, then this
	request will be queued for later execution.

	If <c>wait</c> is set to 1, the function will not return
	until the new data can be added to the SPURS queue.

	If <c>wait</c> is set to 0, the function will return
	immediately, although if there is no space in the SPURS queue, an
	error will be returned.

	For <c>flag,</c> this should be a bit pattern that represents
	this process. Typically you would use <c>1\<\<buffer_number</c>.
	This flag is used to check for the status of the transfer in <c>cellMd5SpuProcessData</c>.
	Note
	For efficiency ensure that the SPURS queue is at least the same
	size as the number of buffers that you will send as well as
	one extra for ending the task.

	The data may not be processed immediately, so the data should
	not be overwritten or changed until the status of this
	operation is checked by calling <c>cellMd5SpuCheckStatus</c>.
	See Also
	<c>cellMd5SpuCreateTask</c>, <c>cellMd5SpuCheckStatus</c>                               */
	int cellMd5SpuProcessData(CellMd5SpuInstance *md5,
		uint64_t data,
		unsigned int size,
		uint16_t flag,
		int wait);

	/* Summary
	Check status of an MD5 process.
	Conditions
	Multi-thread safe.
	Parameters
	md5 :  Pointer to a MD5 SPURS work area.
	flag :    Event flag for the process that is being checked.
	wait :    Waiting mode.
	Returns
	<c>CELL_OK</c>(0) is returned for success.

	If <c>md5==NULL</c>, then <c>CELL_ERROR_HASH_NULL_POINTER</c> is returned.

	If any other error occurs, an error code from <c>cellSpursEventFlagWait</c>
	(if <c>wait==1</c>) or <c>cellSpursEventFlagTryWait</c> (if <c>wait==0</c>)
	is returned.
	Description
	This function returns the status of an MD5 SPURS process
	that was initiated using <c>cellMd5SpuProcessData</c>. If <c>wait</c>
	is set to 1, the function will not return until the process
	that is being checked completes. If <c>wait</c> is set to 0,
	the function will return immediately, with return values as
	specified in <c>cellSpursEventFlagTryWait</c>.
	See Also
	<c>cellMd5SpuCreateTask</c>, <c>cellMd5SpuProcessData</c>             */
	int cellMd5SpuCheckStatus(CellMd5SpuInstance *md5,
		uint16_t flag,
		int wait);


	/* Summary
	End a MD5 SPURS task.[deprecated function maintained for compatibility]
	Conditions
	Multi-thread safe.
	Parameters
	md5 :  Pointer to a MD5 SPURS work area
	hash :  Effective address to store the hash value for the data on
	the PPU. This area should be at least 16 bytes in size
	and 16\-byte aligned.
	Returns
	<c>CELL_OK</c>(0) is returned for success.

	If <c>md5==NULL</c> or <c>hash==0</c>, 
	then <c>CELL_ERROR_HASH_NULL_POINTER</c> is returned.

	<c>CELL_ERROR_HASH_INVALID_ALIGN</c> is returned if <c>hash</c> is not aligned
	to a 16-byte boundary.

	If any other error occurs, an error code from <c>cellSpursQueuePush</c>
	or <c>cellSpursEventFlagWait</c> is returned.

	Description
	This function ends the MD5 SPURS as stored in the <c>md5</c>
	parameter. It will wait for all processes in the SPURS queue to
	complete before updating the value in <c>hash</c> and
	terminating the SPURS task.
	Note
	This function is blocking. For maximum efficiency you should
	ensure all previous data is processed by calling <c>cellMd5SpuCheckStatus</c>
	before calling this function.
	See Also
	<c>cellMd5SpuCreateTask</c>, <c>cellMd5SpuProcessData</c>,
	<c>cellMd5SpuCheckStatus</c>                                                  */
	int cellMd5SpuEndTask(CellMd5SpuInstance *md5,
		uint64_t hash);

	/* Summary
	End a MD5 SPURS task.
	Conditions
	Multi-thread safe.
	Parameters
	md5 :  Pointer to a MD5 SPURS work area
	hash :  Effective address to store the hash value for the data on
	the PPU. This area should be at least 32 bytes in size
	and 16\-byte aligned.
	Returns
	<c>CELL_OK</c>(0) is returned for success.

	If <c>md5==NULL</c> or <c>hash==0</c>, 
	then <c>CELL_ERROR_HASH_NULL_POINTER</c> is returned.

	<c>CELL_ERROR_HASH_INVALID_ALIGN</c> is returned if <c>hash</c> is not aligned
	to a 16-byte boundary.     

	If any other error occurs, an error code from <c>cellSpursQueuePush</c>
	or <c>cellSpursEventFlagWait</c> is returned.

	Description
	This function ends the MD5 SPURS as stored in the <c>md5</c>
	parameter. It will wait for all processes in the SPURS queue to
	complete before updating the value in <c>hash</c> and
	terminating the SPURS task.
	Note
	This function is blocking. For maximum efficiency you should
	ensure all previous data is processed by calling <c>cellMd5SpuCheckStatus</c>
	before calling this function.
	See Also
	<c>cellMd5SpuCreateTask2</c>, <c>cellMd5SpuProcessData</c>,
	<c>cellMd5SpuCheckStatus</c>                                                  */
	int cellMd5SpuEndTask2(CellMd5SpuInstance *md5,
		uint64_t hash);


	/* Summary
	Creates  MD5 SPURS tasks to handle concurrent streaming data .
	Conditions
	Multi-thread safe.
	Parameters
	md5		:		Pointer to a MD5 streams work area.
	taskset		:		Pointer to a SPURS task set instance.
	num_stream	:		Number of data streams to processed concurrently
						<c>(Max 64 supported on a single thread or max 32 supported on 
							 2 separate PPU thread performing the same operation 
							 concurrently)</c>
	Returns
	<table>
	Macro                                   Value        \Description
	-------------------------------------   -----------  --------------------------
	CELL_OK                                 0x00000000   Function returned successfully
	CELL_ERROR_HASH_NULL_POINTER            0x80f00001   <c>taskset</c> or <c>md5</c>
														 is a null pointer.
	CELL_ERROR_HASH_QUEUE_CREATE_FAILED     0x80f00002   SPURS event queue could not be created, 
														 or the event queue could not be 
														 attached to a SPURS queue.
	CELL_ERROR_HASH_FLAG_CREATE_FAILED      0x80f00003   SPURS event flag could not be created, 
														 or the event flag could not be attached
														 to a SPURS queue.
	CELL_ERROR_HASH_INVALID_NOMEM           0x80f00005   Insufficient memory.
	CELL_ERROR_HASH_TASK_CREATE_FAILED      0x80f00006   Failed to create SPURS task.
	CELL_ERROR_HASH_INVALID_STREAM_ID		0x80f00009	 Invalid stream ID.<c>num_stream</c> is not 
														 within a valid range
														
	</table>


	Description
	This function creates multiple SPURS task for calculating the
	MD5 hash value for concurrent data streams. For <c>md5</c> and <c>taskset</c>,
	these should be provided to the function and allocated and
	initialized accordingly.

	The structure <c>md5</c> is used to hold information about
	the MD5 concurrent data streams and  SPURS task associated with it, that is created by this function and
	should not be altered.
	See Also
	<c>cellMd5SpuTerminateStreamingTasks</c>								*/
		int cellMd5SpuCreateStreamingTasks(CellMd5SpuStreamsInstance *md5,
										CellSpursTaskset2 *taskset,
										uint32_t num_stream);


	/* Summary
	Process a chunk of data for a specific stream of data.
	Conditions
	Multi-thread safe.
	Parameters
	md5  :	Pointer to a MD5 streams work area.
	data :		Effective address of data to be processed on the PPU
				(16\-byte aligned)
	size :		Number of bytes to process
	stream_id:	ID of the stream whose data  is going to be processed <c>(Range:0-63)</c>
	flag :		Event flag which is set once processing is complete
	wait :		Waiting mode
	
	Returns
	<c>CELL_OK</c>(0) is returned for success.

	If <c>stream_id==NULL</c> or <c>data==0</c>, then 
	<c>CELL_ERROR_HASH_NULL_POINTER</c> is returned.

	If <c>stream_id</c> is not within the specified range, then 
	<c>CELL_ERROR_HASH_INVALID_STREAM_ID</c> is returned.

	If any other error occurs, an error code from <c>cellSpursQueuePush</c>
	(if <c>wait==1</c>) or cell <c>SpursQueueTryPush</c> (if <c>wait==0</c>)
	is returned.
	
	Description
	This function adds more data to the MD5 SPURS task for a given stream to process.
	If the task is already processing data for a given stream, then this
	request will be queued for later execution.

	If <c>wait</c> is set to 1, the function will not return
	until the new data can be added to the SPURS queue for a given stream task.

	If <c>wait</c> is set to 0, the function will return
	immediately, although if there is no space in the SPURS queue, an
	error will be returned.

	For <c>flag,</c> this should be a bit pattern that represents
	this process. Typically you would use <c>1\<\<buffer_number</c>.
	This flag is used to check for the status of the transfer in <c>cellMd5SpuProcessStreamData</c>.

	The data may not be processed immediately, so the data should
	not be overwritten or changed until the status of this
	operation is checked by calling <c>cellMd5SpuCheckStreamStatus</c>.
	See Also
	<c>cellMd5SpuCreateStreamingTasks</c>, <c>cellMd5SpuGetStreamResult</c> ,<c>cellMd5SpuCheckStreamStatus</c>.                             */
	int cellMd5SpuProcessStreamData(CellMd5SpuStreamsInstance *md5,
											uint64_t data,
											unsigned int size,
											uint64_t stream_id,
											uint16_t flag,
											int wait);

	/* Summary
	Check status of an MD5 process for a specified stream.
	Conditions
	Multi-thread safe.
	Parameters
	md5 :	Pointer to a MD5 streams work area.
	flag :		Event flag for the process that is being checked.
	stream_id :	ID of the stream whose status is to be checked <c>(Range:0-63)</c>
	wait :		Waiting mode.
	Returns
	<c>CELL_OK</c>(0) is returned for success.

	If <c>md5==NULL</c>, then <c>CELL_ERROR_HASH_NULL_POINTER</c> is returned.
	

	If <c>stream_id</c> is not within the specified range, then 
	<c>CELL_ERROR_HASH_INVALID_STREAM_ID</c> is returned.

	If any other error occurs, an error code from <c>cellSpursEventFlagWait</c>
	(if <c>wait==1</c>) or <c>cellSpursEventFlagTryWait</c> (if <c>wait==0</c>)
	is returned.

	Description
	This function returns the status of an MD5 SPURS process
	that was initiated using <c>cellMd5SpuProcessStreamData</c>. If <c>wait</c>
	is set to 1, the function will not return until the process of a given stream 
	that is being checked completes. If <c>wait</c> is set to 0,
	the function will return immediately, with return values as
	specified in <c>cellSpursEventFlagTryWait</c>.
	See Also
	<c>cellMd5SpuCreateStreamingTasks</c>, <c>cellMd5SpuProcessStreamData</c>           */
	int cellMd5SpuCheckStreamStatus(CellMd5SpuStreamsInstance *md5,
										uint16_t flag,
										uint64_t stream_id,
										int wait);

	/* Summary
	Returns the calculated hash value for a specified data stream.
	Conditions
	Multi-thread safe.
	Parameters
	md5			:	Pointer to a MD5 streams work area
	stream_id		:	ID of the stream whose results are calculated  <c>(Range:0-63)</c>
	hash			:	Effective address to store the hash value for the data on
						the PPU. This area should be at least 32 bytes in size
						and 16\-byte aligned.
	Returns
	<c>CELL_OK</c>(0) is returned for success.

	If <c>md5==NULL</c> or <c>hash==0</c>, 
	then <c>CELL_ERROR_HASH_NULL_POINTER</c> is returned.

	If <c>stream_id</c> is not within the specified range, then 
	<c>CELL_ERROR_HASH_INVALID_STREAM_ID</c> is returned.

	<c>CELL_ERROR_HASH_INVALID_ALIGN</c> is returned if <c>hash</c> is not aligned
	to a 16-byte boundary.     

	If any other error occurs, an error code from <c>cellSpursQueuePush</c>
	or <c>cellSpursEventFlagWait</c> is returned.

	Description
	This function calculates the hash value for a specified streams.
	It will wait for all processes in the SPURS queue for a given stream to
	complete before updating the value in <c>hash</c>.The task is not terminated till 
	<c>cellMd5SpuTerminateStreamingTasks </c>. 
	Note
	This function is blocking. For maximum efficiency you should
	ensure all previous data is processed by calling <c>cellMd5SpuCheckStreamStatus</c>
	before calling this function.
	See Also
	<c>cellMd5SpuCreateStreamingTasks</c>, <c>cellMd5SpuProcessStreamData</c>,
	<c>cellMd5SpuCheckStreamStatus</c> ,<c>cellMd5SpuTerminateStreamingTasks </c>*/
	int cellMd5SpuGetStreamResult(CellMd5SpuStreamsInstance* md5,
										  uint64_t stream_id,
										  uint64_t hash);

	int cellMd5SpuGetStreamTaskStatus(CellMd5SpuStreamsInstance* md5,
										uint64_t stream_id,
										int* idle);

	/* Summary
	Terminates the MD5 SPURS tasks to handle concurrent streaming data .
	Conditions
	Multi-thread safe.
	Parameters
	md5 :  Pointer to a MD5 SPURS work area

	Returns
	<c>CELL_OK</c>(0) is returned for success.

	If <c>md5==NULL</c>
	then <c>CELL_ERROR_HASH_NULL_POINTER</c> is returned.


	If any other error occurs, an error code from <c>cellSpursQueuePush</c>
	or <c>cellSpursEventFlagWait</c> is returned.

	Description
	This function ends the MD5 SPURS tasks for concurrent data stream 
	as stored in the <c>md5</c>
	parameter. It will wait for all tasks to terminate which were created 
	Note: 
	Failing to call this function after getting the hash values for all the streams
	will not terminate the SPURS  task which were created by 
	<c>cellMd5SpuCreateStreamingTasks</c>.
	Before calling this function make sure that the results of all the streams are obtained 
	using <c>cellMd5SpuGetStreamResult</c>,failing to do so will terminate all the task for
	concurrent data streams and it will not be possible to get the coputed hash digest for the
	streams.
	See Also
	<c>cellMd5SpuCreateStreamingTasks</c>, <c>cellMd5SpuProcessStreamData</c>,
	<c>cellMd5SpuCheckStreamStatus</c>  <c>cellMd5SpuGetStreamResult </c>*/
	int cellMd5SpuTerminateStreamingTasks(CellMd5SpuStreamsInstance *md5);

#if defined(_LANGUAGE_C_PLUS_PLUS)||defined(__cplusplus)||defined(c_plusplus)
}
#endif	/* defined(_LANGUAGE_C_PLUS_PLUS)||defined(__cplusplus)||defined(c_plusplus) */

#endif	/* _CELL_LIBMD5_SPU_H_ */
