/* SCE CONFIDENTIAL
 PlayStation(R)3 Programmer Tool Runtime Library 475.001
 * Copyright (C) 2010 Sony Computer Entertainment Inc.
 * All Rights Reserved.
 */
#ifndef __CELL_GEM_LIBGEM_H__
#define __CELL_GEM_LIBGEM_H__

#include <stdint.h>
#include <sys/types.h>
#include <cell/spurs.h>
#ifndef __SPU__
#include <sys/paths.h>
#include <vec_types.h>
#endif

/**
 * @defgroup Group11Other Miscellaneous Defines
 * @brief 
 * Miscellaneous Defines 
 * @{
 */
#define CELL_GEM_VERSION    2  ///< Used for API backwards compatibility.

#define CELL_GEM_MAX_NUM             (4)       ///< Maximum number of motion controllers
#define CELL_GEM_NOT_CALIBRATED      8         ///< The specified motion controller's magnetometer has not been calibrated in XMB, so magnetometer will not be used.
#define CELL_GEM_SPHERE_RADIUS_MM    22.5f     ///< Radius of the physical sphere
#define CELL_GEM_LATENCY_OFFSET      (-22000)  ///< Average time offset (in microseconds) of sensing. Use for <c>cellGemGetState()</c> queries using <c>CELL_GEM_STATE_FLAG_CURRENT_TIME</c>.

/** @} */ // end of GroupOther group

/**
 * @defgroup Group02CellGemInfo Defines for CellGemInfo
 * @brief 
 * <c>CellGemInfo</c> status flags. 
 * @{
 */
#define CELL_GEM_STATUS_DISCONNECTED    (0)  ///< Motion controller is disconnected
#define CELL_GEM_STATUS_READY           (1)  ///< Motion controller is connected and ready


/** @} */ // end of GroupCellGemInfo group
/**
 * @brief Structure containing motion controller connection information.
 *
 * This structure is used by the <c>cellGemGetInfo()</c> function.
 * The PlayStation&reg;3 has 7 ports for all types of controllers.  They are normally 
 * assigned sequentially starting with 1 first, but for the motion controller the 
 * ports are assigned in reverse sequential order starting with 7.
 *
 * The "gem slot" relates to the <c><i>gem_num</i></c> argument that is in so many libgem functions.  
 * It is the logical numbering of the motion controllers, starting with 0 and assigned sequentially.
 * Typically, motion controller 0 is assigned port 7 of the PlayStation&reg;3;  motion controller 1
 * is assigned port 6, and so on.
 * 
 * libgem only uses the <c><i>gem_num</i></c> as an argument, but the <c>CellGemInfo</c> structure lets you 
 * find out what the PlayStation&reg;3 port number is (the controller number in the system menu displayed
 * by pressing the PS button).
 *
 * @see 
 * <c>cellGemGetInfo</c>
*/
typedef struct CellGemInfo
{
    uint32_t max_connect;               ///< The number of motion controllers that can be connected and active, as specified by <c>cellGemInit()</c>.
    uint32_t now_connect;               ///< The number of motion controllers currently connected and active.
    uint32_t status[CELL_GEM_MAX_NUM];  ///< Connection status (<c>CELL_GEM_STATUS_DISCONNECTED</c> or <c>CELL_GEM_STATUS_READY</c>) for each gem slot (0 - 3).
    uint32_t port[CELL_GEM_MAX_NUM];    ///< libpad port number (1-7) assigned to each motion controller; assignment begins with 7 and counts down.  Equals 0 when disconnected.
} CellGemInfo;



/**
 * @defgroup Group03CellGemPadData Defines for bit assignment of digital buttons
 * @brief
 * <c>CellGemPadData</c> defines for bit assignment of digital buttons.
 * @{
 */
#define CELL_GEM_CTRL_SQUARE      (1 << 7)  ///< Square button
#define CELL_GEM_CTRL_CROSS       (1 << 6)  ///< Cross button
#define CELL_GEM_CTRL_CIRCLE      (1 << 5)  ///< Circle button
#define CELL_GEM_CTRL_TRIANGLE    (1 << 4)  ///< Triangle button
#define CELL_GEM_CTRL_MOVE        (1 << 2)  ///< Move button
#define CELL_GEM_CTRL_T           (1 << 1)  ///< T button
#define CELL_GEM_CTRL_START       (1 << 3)  ///< START button
#define CELL_GEM_CTRL_SELECT      (1 << 0)  ///< SELECT button
/** @} */ // end of GroupCellGemPadData group


/**
 * @defgroup Group01CellGemExtPortData Defines for bit assignment of digital buttons
 * @brief
 * Bit assignments for <c>CellGemExtPortData</c> <c><i>status</i></c> member.
 * @{
 */


#define CELL_GEM_EXT_CONNECTED    (1 << 0)  ///< External port connectivity status. 1 indicates a connection; 0 indicates no connection.
#define CELL_GEM_EXT_EXT0         (1 << 1)  ///< Pre-defined external port output bit
#define CELL_GEM_EXT_EXT1         (1 << 2)  ///< Pre-defined external port output bit

/** @} */ // end of GroupCellGemExtPortData group

/**
 * @defgroup Group11ExternalPort Values pretaining to the extention connector
 * @brief Values used to describe characteristics of the extension connector.
 * @{
 */
#define CELL_GEM_EXTERNAL_PORT_DEVICE_INFO_SIZE 38  ///< Array size for <c>cellGemReadExternalPortDeviceInfo()</c>
#define CELL_GEM_EXTERNAL_PORT_OUTPUT_SIZE 40       ///< Array size for <c>cellGemWriteExternalPort()</c>
#define CELL_GEM_NO_EXTERNAL_PORT_DEVICE 9          ///< Return value from <c>cellGemReadExternalPortDeviceInfo()</c> indicating that the specified motion controller has not established a connection to the PlayStation&reg;3.
/** @} */ // end of Group11ExternalPort group

/**
 * @defgroup Group09CellGemPrepareCamera Limits for cellGemPrepareCamera max_exposure argument
 * @brief Limits for <c>cellGemPrepareCamera()</c> <c><i>max_exposure</i></c> argument.
 * @{
 */
#define CELL_GEM_MIN_CAMERA_EXPOSURE 40   ///< Minimum camera exposure that can be used by libgem.
#define CELL_GEM_MAX_CAMERA_EXPOSURE 511  ///< Maximum camera exposure that can be used by libgem.

/** @} */ // end of GroupcellGemPrepareCamera group


/**
 * @defgroup Group05CellGemGetState Flags for cellGemGetState
 * @brief Flags for <c>cellGemGetState()</c>.
 * @{
 */


#define CELL_GEM_STATE_FLAG_CURRENT_TIME 0       ///< State timing matched to the current system time when <c>cellGemGetState()</c> is called.
#define CELL_GEM_STATE_FLAG_LATEST_IMAGE_TIME 1  ///< State timing matched to latest camera image prior to when <c>cellGemGetState()</c> is called.
#define CELL_GEM_STATE_FLAG_TIMESTAMP 2          ///< State timing matched to a particular system timestamp.

/** @} */ // end of GroupcellGemGetState group



/**
 * @defgroup Group04CellGemGetInertialState Flags for cellGemGetState
 * @brief Flags for <c>cellGemGetInertialState()</c>.
 * @{
 */

#define CELL_GEM_INERTIAL_STATE_FLAG_LATEST 0    ///< Retrieve latest available inertial state when <c>cellGemGetInertialState()</c> is called.
#define CELL_GEM_INERTIAL_STATE_FLAG_PREVIOUS 1  ///< Retrieve inertial data that precedes the specified system timestamp.
#define CELL_GEM_INERTIAL_STATE_FLAG_NEXT 2      ///< Retrieve inertial data that is subsequent to the specified system timestamp.
/** @} */ // end of GroupcellGemGetState group



/**
 * @defgroup Group07CellGemGetStateRC cellGemGetState return codes
 * @brief Return codes for <c>cellGemGetState()</c>.
 * @{
 */


#define CELL_GEM_NOT_CONNECTED 1               ///< The specified motion controller has not established a connection to the PlayStation&reg;3.
#define CELL_GEM_SPHERE_NOT_CALIBRATED 2       ///< The sphere of the specified motion controller is not calibrated.
#define CELL_GEM_SPHERE_CALIBRATING 3          ///< The sphere of the specified motion controller is in the process of calibrating.
#define CELL_GEM_COMPUTING_AVAILABLE_COLORS 4  ///< libgem is determining the available tracking hues for the sphere LED.
#define CELL_GEM_HUE_NOT_SET 5                 ///< The hue of the sphere is not set.
#define CELL_GEM_NO_VIDEO 6                    ///< No usable video is specified.
#define CELL_GEM_TIME_OUT_OF_RANGE 7           ///< No state is available at the specified time
/** @} */ // end of GroupcellGemGetState group


/**
 * @defgroup Group10CellGemTrackHues Special values for cellGemTrackHues
 * @brief Special values for <c>cellGemTrackHues()</c>.
 * @{
 */

#define CELL_GEM_DONT_TRACK_HUE     (2<<24)  ///< A value signaling to libgem to stop tracking a controller
#define CELL_GEM_DONT_CARE_HUE      (4<<24)  ///< A value signaling to libgem to pick the best available tracking hue
#define CELL_GEM_DONT_CHANGE_HUE    (8<<24)  ///< To be supported (currently treated as <c>CELL_GEM_DONT_CARE_HUE</c>)
/** @} */ // end of GroupcellGemTrackHues group



/**
 * @defgroup Group08CellGemGetStatusFlags Masks for cellGemGetStatusFlags
 * @brief Masks for <c>cellGemGetStatusFlags()</c>.
 * @{
 */


#define CELL_GEM_FLAG_CALIBRATION_OCCURRED                      (1 << 0)   ///< Status flag indicating that calibration occurred; if this flag is returned in the status, look for a calibration error flag as well.
#define CELL_GEM_FLAG_CALIBRATION_SUCCEEDED                     (1 << 1)   ///< Status flag indicating that calibration occurred and succeeded.
#define CELL_GEM_FLAG_CALIBRATION_FAILED_CANT_FIND_SPHERE       (1 << 2)   ///< Status flag indicating that the most recent calibration failed because the sphere could not be found in the camera image.
#define CELL_GEM_FLAG_CALIBRATION_FAILED_MOTION_DETECTED        (1 << 3)   ///< Status flag indicating that the most recent calibration failed because the sphere was not held steady for the calibration interval.
#define CELL_GEM_FLAG_CALIBRATION_FAILED_BRIGHT_LIGHTING        (1 << 4)   ///< Status flag indicating that the most recent calibration failed because the sphere light could not be reliably distinguished from the ambient light.
#define CELL_GEM_FLAG_CALIBRATION_WARNING_MOTION_DETECTED       (1 << 5)   ///< Status flag indicating that the most recent calibration was not ideal because the sphere was not held steady during part of the calibration.
#define CELL_GEM_FLAG_CALIBRATION_WARNING_BRIGHT_LIGHTING       (1 << 6)   ///< Status flag indicating that the most recent calibration was not ideal because the sphere light is not well distinguished from the ambient light.
#define CELL_GEM_FLAG_LIGHTING_CHANGED                          (1 << 7)   ///< To be supported.
#define CELL_GEM_FLAG_WRONG_FIELD_OF_VIEW_SETTING               (1 << 8)   ///< To be supported.
#define CELL_GEM_FLAG_CAMERA_PITCH_ANGLE_CHANGED                (1 << 9)   ///< To be supported.
#define CELL_GEM_FLAG_VARIABLE_MAGNETIC_FIELD                   (1 << 10)  ///< To be supported.
#define CELL_GEM_FLAG_WEAK_MAGNETIC_FIELD                       (1 << 11)  ///< To be supported.
#define CELL_GEM_FLAG_VERY_COLORFUL_ENVIRONMENT                 (1 << 12)  ///< Status flag indicating that the user's room is very colorful and may interfere with tracking.
#define CELL_GEM_FLAG_CURRENT_HUE_CONFLICTS_WITH_ENVIRONMENT    (1 << 13)  ///< Status flag indicating that the current set hue is conflicting with something in the scene.
#define CELL_GEM_ALL_FLAGS                                      0xffffffffffffffffull   ///< Represents all flags for <c>cellGemClearStatusFlags()</c>.
/** @} */ // end of GroupcellGemGetStatusFlags group



/**
 * @defgroup Group06CellGemGetStateTrackingFlags Masks for CellGemState tracking_flags
 * @brief Masks for <c>CellGemState</c> <c><i>tracking_flags</i></c> member.
 * @{
 */
#define CELL_GEM_TRACKING_FLAG_POSITION_TRACKED    (1 << 0)  ///< library has sufficient image data to track position
#define CELL_GEM_TRACKING_FLAG_VISIBLE             (1 << 1)  ///< sphere is visible in image data preceding this state
/** @} */ // end of GroupcellGemGetStateTrackingFlags group


/* Error codes */
#define CELL_GEM_ERROR_RESOURCE_ALLOCATION_FAILED   CELL_ERROR_CAST(0x80121801)  ///< libgem initialization failed to allocate the required resources.
#define CELL_GEM_ERROR_ALREADY_INITIALIZED          CELL_ERROR_CAST(0x80121802)  ///< libgem initialization failed as the library is already initialized.
#define CELL_GEM_ERROR_UNINITIALIZED                CELL_ERROR_CAST(0x80121803)  ///< Call failed because libgem is not initialized.
#define CELL_GEM_ERROR_INVALID_PARAMETER            CELL_ERROR_CAST(0x80121804)  ///< Call failed because one or more parameters passed was not valid.
#define CELL_GEM_ERROR_INVALID_ALIGNMENT            CELL_ERROR_CAST(0x80121805)  ///< Call failed because the video buffer could not be read.
#define CELL_GEM_ERROR_UPDATE_NOT_FINISHED          CELL_ERROR_CAST(0x80121806)  ///< <c>cellGemUpdateStart()</c> call failed because it was called before <c>cellGemUpdateFinish()</c> completed.
#define CELL_GEM_ERROR_UPDATE_NOT_STARTED           CELL_ERROR_CAST(0x80121807)  ///< <c>cellGemUpdateFinish()</c> call failed because it was called before <c>cellGemUpdateStart()</c>.
#define CELL_GEM_ERROR_CONVERT_NOT_FINISHED         CELL_ERROR_CAST(0x80121808)  ///< <c>cellGemConvertVideoStart()</c> call failed because it was called before <c>cellGemConvertVideoFinish()</c>.
#define CELL_GEM_ERROR_CONVERT_NOT_STARTED          CELL_ERROR_CAST(0x80121809)  ///< <c>cellGemConvertVideoFinish()</c> call failed because it was called before <c>cellGemConvertVideoStart()</c>. 
#define CELL_GEM_ERROR_WRITE_NOT_FINISHED           CELL_ERROR_CAST(0x8012180A)  ///< <c>cellGemWriteExternalPort()</c> call failed because it was called before previous <c>cellGemWriteExternalPort()</c> completed.
#define CELL_GEM_ERROR_NOT_A_HUE                    CELL_ERROR_CAST(0x8012180B)  ///< Returned by <c>cellGemGetTrackerHue()</c> if the controller is not currently being tracked.

/** 
 * @brief Structure containing motion controller keypad state information.
 *
 * This data is retrieved by <c>cellGemGetState()</c> and <c>cellGemGetInertialState()</c>.
 *
 * @see 
 * <c>CellGemState</c>, <c>CellGemInertialState</c>
*/
typedef struct CellGemPadData{
	uint16_t digitalbuttons;  ///< State of each of the button controls. See <c>CELL_GEM_CTRL_CIRCLE</c>, <c>CELL_GEM_CTRL_CROSS</c>, <c>CELL_GEM_CTRL_MOVE</c>, <c>CELL_GEM_CTRL_SELECT</c>, <c>CELL_GEM_CTRL_SQUARE</c>, <c>CELL_GEM_CTRL_START</c>, <c>CELL_GEM_CTRL_T</c>, and <c>CELL_GEM_CTRL_TRIANGLE</c>.
	uint16_t analog_T;  ///< Analog value of T button
} CellGemPadData;

/** 
 * @brief Structure containing external port information.
 *
 * This data is retrieved by <c>cellGemGetState()</c> and <c>cellGemGetInertialState()</c>.
 * It relates to external devices connected to the motion controller through the extension connector (not the USB connector).
 * Predefined data definitions and packet space exist for the directional buttons, digital triggers,
 * and analog sticks that are found on the SIXAXIS&trade; and DUALSHOCK&reg;3 wireless controllers.  In addition, a custom data area exists for data
 * unique to particular hardware attachments.
 *
 * @see 
 * <c>CellGemState</c>, <c>CellGemInertialState</c>, <c>cellGemGetState</c>, <c>cellGemGetInertialState</c>
*/
typedef struct CellGemExtPortData {
	uint16_t status;          ///< Port connection status, including <c>CELL_GEM_EXT_CONNECTED</c>, <c>CELL_GEM_EXT_EXT0</c>, and <c>CELL_GEM_EXT_EXT1</c>.
	uint16_t digital1;        ///< Directional button bits and L3, R3 buttons. Identical to libpad digital button <c>CELL_PAD_BTN_OFFSET_DIGITAL1</c>.
	uint16_t digital2;        ///< L1, L2, R1, R2 button bits. Identical to libpad digital button <c>CELL_PAD_BTN_OFFSET_DIGITAL2</c>.
	uint16_t analog_right_x;  ///< Right stick horizontal data.
	uint16_t analog_right_y;  ///< Right stick vertical data.
	uint16_t analog_left_x;   ///< Left stick horizontal data.
	uint16_t analog_left_y;   ///< Left stick vertical data.
	unsigned char custom[5];  ///< Custom data area into which product-specific external port data is placed.
} CellGemExtPortData;

/** 
 * @brief The state of the motion controller as computed from video and inertial sensors.
 *
 * Structure containing the state of the motion controller as computed from video and inertial sensors. 
 * The elements are computed in &#8220;world&#8221; coordinates as distinct from camera or motion controller coordinates.
 * Both world and camera coordinate systems have their origin at the camera&#8216;s center of projection.
 * 
 * Camera coordinates are defined as:
 * <ul><li>x axis increases to the right as the user faces the camera.</li>
 * <li>y axis increases up in the image plane.</li>
 * <li>z axis is the direction through the camera&#8216;s optical center, and increases towards 
 * the user (away from the camera).</li></ul>
 * 
 * World coordinates are offset from camera coordinates by the pitch angle of the camera; 
 * they are defined as:
 * <ul><li>x axis increases right as the user faces the camera.</li>
 * <li>y axis increases up, parallel to gravity.</li>
 * <li>z axis is the direction through the camera&#8216;s optical center projected onto the ground,
 * and increases towards the user (away from the camera).</li></ul>
 * 
 * The motion controller coordinate system has its origin at the center of the sphere, with coordinates as follows: 
 * <ul><li>y axis increases up from the button face.</li>
 * <li>z axis increasing from the center of the sphere towards and aligned with the base of the handle.</li>
 * <li>x axis increasing right perpendicular from the button face.</li></ul>
 *
 * @see 
 * <c>CellGemPadData</c>, <c>cellGemGetState</c>
*/
typedef struct CellGemState {
    vec_float4 pos;           ///< Position of the center of the sphere in world coordinates (mm).
    vec_float4 vel;           ///< Velocity of sphere in world coordinates (mm/s).
    vec_float4 accel;         ///< Acceleration of sphere in world coordinates (mm/s^2).
    vec_float4 quat;          ///< Quaternion orientation of the motion controller in world (<c>x</c>,<c>y</c>,<c>z</c>,<c>w</c>).
    vec_float4 angvel;        ///< Angular velocity of the motion controller in world coordinates (radians/s).
    vec_float4 angaccel;      ///< Angular acceleration of the motion controller in world coordinates (radians/s^2).
    vec_float4 handle_pos;    ///< Position near the player&#8216;s hand in world coordinates (mm).
    vec_float4 handle_vel;    ///< Velocity near the player&#8216;s hand in world coordinates (mm/s).
    vec_float4 handle_accel;  ///< Acceleration near the player&#8216;s hand in world coordinates (mm/s^2).
    CellGemPadData pad;       ///< State of the motion controller button controls.
    CellGemExtPortData ext;   ///< External port data as described by <c>CellGemExtPortData</c>.
    system_time_t timestamp;  ///< System timestamp.
    float temperature;        ///< The temperature of the sensor (used internally).
    float camera_pitch_angle; ///< The camera pitch angle that was used for computing this state.
    uint32_t tracking_flags;  ///< Flags for tracking status, including: <c>CELL_GEM_TRACKING_FLAG_POSITION_TRACKED</c> and <c>CELL_GEM_TRACKING_FLAG_VISIBLE</c>.
} CellGemState;


/** 
 * @brief Inertial sensor data.
 *
 * Structure containing the data returned from the inertia sensors. The sensor readings are in sensor-local coordinates.
 * This data is raw data from the sensor in the handle of the motion controller. If you use this data instead of the motion controller
 * positional data, you need to perform the appropriate transformations to adjust the position from the sensor to the
 * center of the sphere. 
 * @note
 * We recommend that you use the motion controller coordinate data from the <c>CellGemState</c> structure rather than this sensor data.
 * @see
 * <c>CellGemState</c>, <c>cellGemGetInertialState</c>
*/
typedef struct CellGemInertialState {
	vec_float4 accelerometer;       ///< Accelerometer readings in sensor coordinates (G units).
	vec_float4 gyro;                ///< Gyro readings in sensor coordinates (radians/s).
	vec_float4 accelerometer_bias;  ///< Estimated accelerometer biases in sensor coordinates (G units). Subtract from accelerometer readings to obtain raw readings. 
	vec_float4 gyro_bias;           ///< Estimated gyro biases in sensor coordinates (radians/s). Subtract from gyro readings to obtain raw readings.
	CellGemPadData pad;             ///< State of the motion controller buttons.
	CellGemExtPortData ext;         ///< External port data as described by <c>CellGemExtPortData</c>.
	system_time_t timestamp;        ///< System timestamp.
	int counter;                    ///< Numbers the sensor data sequentially.
	float temperature;              ///< The temperature of the sensor (used internally). 
} CellGemInertialState;


/** 
 * @brief Computer vision data.
 *
 * Structure containing the data returned from the computer camera.
 *
 * The sphere coordinates are calculated with the origin in the upper left of image 
 * with <c><i>u</i></c> increasing to the right and <c><i>v</i></c> increasing down.
 * 
 * The projection coordinates are calculated with the origin at the center of the image 
 * with <c><i>x</i></c> increasing right and <c><i>y</i></c> increasing up. The 3D position 
 * of the sphere in the camera 
 * coordinate system is given by:
 * 
 * <c>(<i>projectionx</i> * <i>distance</i>, <i>projectiony</i> * <i>distance</i>, <i>distance</i>)</c>
 * 
 * @see
 * <c>cellGemGetImageState</c>, <c>cellCameraReadEx</c> (see the <i>libcamera Reference</i>)
*/
typedef struct CellGemImageState {
	system_time_t frame_timestamp;  ///< Timestamp returned by <c>cellCameraReadEx()</c>.
	system_time_t timestamp;        ///< Timestamp for when the sphere was actually imaged; includes rolling shutter and exposure adjustments.
	float u;                        ///< Horizontal pixel coordinate of center of the sphere.
	float v;                        ///< Vertical pixel coordinate of center of the sphere.
	float r;                        ///< Radius of sphere in pixels.
	float projectionx;              ///< Horizontal normalized image coordinate.
	float projectiony;              ///< Vertical normalized image coordinate.
	float distance;                 ///< Distance from the plane located at the center of projection and parallel to the image plane.
	unsigned char visible;          ///< Whether visible on camera at the time of this state.
	unsigned char r_valid;          ///< Whether or not <c><i>r</i></c> was computed this frame. Set to 0, <c><i>r</i></c> and <c><i>distance</i></c> are data from previous camera frames.
} CellGemImageState;


/** 
 * @brief PlayStation&reg;Eye camera information.
 *
 * Structure containing the state of the camera.
 *
*/
typedef struct CellGemCameraState {
  int exposure;                ///< Camera exposure setting in image rows (same as libcamera setting)
  float exposure_time;         ///< Camera exposure time (in seconds)
  float gain;                  ///< Camera effective gain (minimum is 1.0, max is 4.0)
  float pitch_angle;           ///< Pitch angle used for <c>CellGemState</c> computation
  float pitch_angle_estimate;  ///< Current estimate of pitch angle
} CellGemCameraState;


/** 
 * @brief PlayStation&reg;Move motion controller initialization data.
 *
 * Structure containing the data with which the motion controller is initialized.
 *
 * @see
 * <c>cellGemAttributeInit</c>
*/
typedef struct CellGemAttribute {
	uint32_t version;           ///< Specifies the libgem library version for which this application was built.
	uint32_t max_connect;       ///< Specifies the maximum number of motion controllers (up to <c>CELL_GEM_MAX_NUM</c>).
	void * memory_ptr;          ///< Specifies the memory for libgem.  If NULL, libgem allocates its own memory. Use <c>cellGemGetMemorySize()</c> to determine how much memory to allocate.
	CellSpurs *spurs;           ///< Required. Specifies a pointer to <c>CellSpurs</c>.
	uint8_t spu_priorities[8];  ///< Specifies an array of SPU priorities to use for libgem SPURS workload.
} CellGemAttribute;


/**
 * @brief Options for the <c><i>output_format</i></c> member of the structure <c>CellGemVideoConvertAttribute</c>.
 * 
 * Enumeration of the mode used to manipulate video output.
 * 
 * @see <c>cellGemGetMemorySize</c>, <c>CellGemVideoConvertAttribute</c>
 * 
 */
typedef enum
{
	CELL_GEM_NO_VIDEO_OUTPUT=1,          ///< Disable video output.
	CELL_GEM_RGBA_640x480,               ///< RGBA output; 640*480*4-byte output buffer required.
	CELL_GEM_YUV_640x480,                ///< YUV output; 640*480+640*480+640*480-byte output buffer required (contiguous).
	CELL_GEM_YUV422_640x480,             ///< YUV output; 640*480+320*480+320*480-byte output buffer required (contiguous).
	CELL_GEM_YUV411_640x480,             ///< YUV411 output; 640*480+320*240+320*240-byte output buffer required (contiguous).
	CELL_GEM_RGBA_320x240,               ///< RGBA output; 320*240*4-byte output buffer required.
	CELL_GEM_BAYER_RESTORED,             ///< Bayer pattern output, 640x480, gamma and white balance applied, output buffer required.
	CELL_GEM_BAYER_RESTORED_RGGB,        ///< Restored Bayer output, 2x2 pixels rearranged into 320x240 RG1G2B.
	CELL_GEM_BAYER_RESTORED_RASTERIZED,  ///< Restored Bayer output, R,G1,G2,B rearranged into 4 contiguous 320x240 1-channel rasters.
} CellGemVideoConvertFormatEnum;


/**
 * @brief Flags for the <c><i>conversion_flags</i></c> member of the structure <c>CellGemVideoConvertAttribute</c>. 
 *
 * Enumeration of flags used to manipulate video output.
 * 
 * @see
 * <c>CellGemVideoConvertAttribute</c>
*/
typedef enum {
	CELL_GEM_AUTO_WHITE_BALANCE=0x1,            ///< Dynamically adjust the red, green, and blue gains based on sphere color.
	CELL_GEM_GAMMA_BOOST=0x2,                   ///< Boost the dark areas to improve display.
	CELL_GEM_COMBINE_PREVIOUS_INPUT_FRAME=0x4,  ///< Blend with the previous frame to reduce noise.
	CELL_GEM_FILTER_OUTLIER_PIXELS=0x8,         ///< Filter out "stuck" pixels during conversion.
} CellGemVideoConvertFlagEnum;


/** 
 * @brief Video conversion initialization data.
 *
 * Structure containing the data with which the video conversion mode is initialized.
 *
 * @see
 * <c>CellGemVideoConvertFlagEnum</c>, <c>CellGemVideoConvertFormatEnum</c>, <c>cellGemAttributeInit</c>, 
 * <c>cellGemVideoConvertAttributeInit</c>, <c>cellGemVideoConvertAttributeInitRgba</c>
*/
typedef struct CellGemVideoConvertAttribute {
	int version;           ///< Library version: set to <c>CELL_GEM_VERSION</c>.
	int output_format;     ///< Output format, one of <c>CellGemVideoConvertFormatEnum</c>.
	int conversion_flags;  ///< Logical OR of conversion flags, from <c>CellGemVideoConvertFlagEnum</c>.
	float gain;            ///< Overall gain adjustment.
	float red_gain;        ///< Red gain. Usually <c>1.0f</c> if <c>CELL_GEM_AUTO_WHITE_BALANCE</c> is enabled.
	float green_gain;      ///< Green gain. Usually <c>1.0f</c> if <c>CELL_GEM_AUTO_WHITE_BALANCE</c> is enabled.
	float blue_gain;       ///< Blue gain. Usually <c>1.0f</c> if <c>CELL_GEM_AUTO_WHITE_BALANCE</c> is enabled.
	void *buffer_memory;   ///< Optional. Used for summing multiple Bayer frames to improve image quality.
	void *video_data_out;  ///< Output buffer for video frame. Overwritten between <c>cellGemUpdateStart()</c> and <c>cellGemUpdateFinish()</c>.
	uint8_t alpha;         ///< For RGBA output, this is the alpha value [0,255].
} CellGemVideoConvertAttribute;


#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

/** 
 * @brief Retrieves the amount of memory libgem needs to operate
 *
 * This function returns the amount of memory libgem needs to operate, based on the 
 * maximum number of controllers that the game supports. When calling <c>cellGemInit()</c>,
 * you can provide a pointer to the memory provided by the game for libgem.
 * If the game does not allocate memory, libgem allocates its own memory. 
 * 
 * To control memory allocation, call <c>cellGemGetMemorySize()</c> to find out how 
 * much memory libgem will need, allocate the appropriate memory, include a pointer to the
 * memory in <c>CellGemAttribute</c>, and pass the pointer to <c>cellGemInit()</c>. 
 *
 * <c>cellGemGetMemorySize()</c> is the only function that can be and should be called 
 * before calling <c>cellGemInit()</c>. 
 * 
 * @param max_connect   The maximum number of controllers the game allows. 
                        This value is less than or equal to <c>CELL_GEM_MAX_NUM</c>.
 * 
 * @return
 * <c>CELL_GEM_ERROR_INVALID_PARAMETER</c> if <c><i>max_connect</i></c> is greater than <c>CELL_GEM_MAX_NUM</c>. Otherwise, the size of memory required by libgem.
 * @see
 * <c>CellGemAttribute</c>, <c>cellGemInit</c>
*/
int32_t cellGemGetMemorySize(int max_connect);


/** 
 * @brief Initializes the <c>CellGemAttribute</c> structure.
 *
 * This function initializes the motion controllers by setting values in the <c>CellGemAttribute</c> structure.
 * 
 * @param attr           Pointer to struct to be initialized
 * @param max_connect    The maximum number of motion controllers (up to <c>CELL_GEM_MAX_NUM</c>)
 * @param memory_ptr     The memory for libgem use
 * @param spurs          Required. A pointer to <c>CellSpurs</c>
 * @param spu_priorities Array of SPU priorities to use for libgem SPURS workload
 *
 * @see
 * <c>CellGemAttribute</c>, <c>cellGemInit</c>
*/
static inline void cellGemAttributeInit(CellGemAttribute* attr, 
	uint32_t max_connect,
	void * memory_ptr,																CellSpurs *spurs, 
	const uint8_t spu_priorities[8])
{
	attr->version = CELL_GEM_VERSION;
	attr->max_connect = max_connect;
	attr->spurs = spurs;
	attr->memory_ptr = memory_ptr;
	for(int i=0; i<8; ++i) attr->spu_priorities[i] = spu_priorities[i];
}


/** 
 * @brief Initializes libgem.
 * 
 * This function performs the required initialization for using libgem.
 * 
 * @param attr   Attribute structure initialized with <c>cellGemAttributeInit()</c>
 *
 * @retval CELL_OK Successful completion.
 * @retval CELL_GEM_ERROR_RESOURCE_ALLOCATION_FAILED   Failed to allocate a resource.
 * @retval CELL_GEM_ERROR_INVALID_PARAMETER   One of the parameters is invalid.
 * @retval CELL_GEM_ERROR_ALREADY_INITIALIZED  libgem is already initialized.
 * @see
 * <c>cellGemAttributeInit</c>
*/
int32_t cellGemInit(const CellGemAttribute *attr);


/** 
 * @brief Terminates libgem.
 * 
 * This function ends the operation of libgem, freeing allocated memory.
 * 
 * @retval CELL_OK Successful completion.
 * @retval CELL_GEM_ERROR_UNINITIALIZED   libgem is not initialized.
 *
*/
int32_t cellGemEnd(void);


/** 
 * @brief Prepares the PlayStation&reg;Eye camera for tracking.
 * 
 * This function prepares the PlayStation&reg;Eye camera for libgem tracking. You need to call this function one time after calling
 * <c>cellGemInit()</c> and <c>cellCameraReset()</c> for proper operation of libgem.
 * The camera attributes affected are: <c>AGC</c>, <c>AEC</c>, <c>AWB</c>, <c>GAMMA</c>, and <c>PIXELOUTLIERFILTER</c> (all disabled).
 * In addition, <c>GAIN</c>, <c>EXPOSURE</c>, <c>REDBLUE_GAIN</c>, and <c>GREEN_GAIN</c> are adjusted as needed by libgem.
 *
 * @param   max_exposure		The setting for the maximum allowed exposure (40-511).
					Sets the maximum possible number of image rows of exposure time (511 rows in each camera frame).
					Note that a longer exposure time means decreased image noise but increased motion blur, which 
					has a negative effect on libgem tracking performance. Use <c>CELL_GEM_MIN_CAMERA_EXPOSURE</c> and <c>CELL_GEM_MAX_CAMERA_EXPOSURE</c> to use the minimum and maximum exposure values.
 * @param   image_quality		Adjusts the image quality level (0.0-1.0).  A setting of 1.0 is the best possible quality while 
					0.0 is the best possible tracking. 
 *
 * @retval CELL_OK   Successful completion.
 * @retval EBUSY     Camera preparation or calibration is already in progress.
 * @retval CELL_GEM_ERROR_UNINITIALIZED   If libgem is not initialized.
 * @see
 * <c>cellCameraReset</c> (see the <i>libcamera Reference</i>), <c>cellGemInit</c>
*/
int32_t cellGemPrepareCamera(int max_exposure, float image_quality);


/** 
 * @brief Determines required video output buffer size based on the format of the specified video.
 * 
 * This function determines the required video output buffer size based on video format. 
 * The <c><i>video_data_out</i></c> member of <c>CellGemVideoConvertAttribute</c> structure points to the 
 * video output buffer.
 * 
 * @param output_format Conversion format, one of <c>CellGemVideoConvertFormatEnum</c>.
 *
 * @return
 * -1 if unknown <c><i>output_format</i></c>. Otherwise, the required size of video output buffer.
 *
 * @see
 * <c>CellGemVideoConvertAttribute</c>, <c>CellGemVideoConvertFormatEnum</c>, <c>cellGemVideoConvertAttributeInit</c>
*/
static inline int32_t cellGemGetVideoConvertSize(int output_format)
{
	switch(output_format)
	{
	case CELL_GEM_NO_VIDEO_OUTPUT:
		return 0;
	case CELL_GEM_RGBA_640x480:
		return 640*480*4;
	case CELL_GEM_YUV_640x480:
		return 640*480*3;
	case CELL_GEM_RGBA_320x240:
		return 320*240*4;
	case CELL_GEM_YUV411_640x480:
		return 640*480+320*240+320*240;
	case CELL_GEM_YUV422_640x480:
		return 640*480+320*480+320*480;
	case CELL_GEM_BAYER_RESTORED:
		return 640*480;
	case CELL_GEM_BAYER_RESTORED_RGGB:
		return 320*240*4;
	case CELL_GEM_BAYER_RESTORED_RASTERIZED:
		return 320*240*4;
	default:
		return -1;
	}
}


/** 
 * @brief Initializes the <c>CellGemVideoConvertAttribute</c> structure.
 * 
 * This function initializes the <c>CellGemVideoConvertAttribute</c> structure.
 * 
 * @param retAttr Attribute structure to initialize.
 * @param output_format Output format, one of <c>CellGemVideoConvertFormatEnum</c>.
 * @param conversion_flags Logical OR of conversion flags, from <c>CellGemVideoConvertFlagEnum</c>.
 * @param gain       Overall gain adjustment.
 * @param red_gain   Red gain; Usually <c>1.0f</c> if <c>CELL_GEM_AUTO_WHITE_BALANCE</c> is enabled.
 * @param green_gain Green gain; Usually <c>1.0f</c> if <c>CELL_GEM_AUTO_WHITE_BALANCE</c> is enabled.
 * @param blue_gain  Blue gain; Usually <c>1.0f</c> if <c>CELL_GEM_AUTO_WHITE_BALANCE</c> is enabled.
 * @param buffer_memory  Optional. Used for summing multiple Bayer frames to improve image quality.
 * @param video_data_out  Output buffer for video frame. Overwritten between <c>cellGemUpdateStart()</c> and <c>cellGemUpdateFinish()</c>.
 * @param alpha  For RGBA output, this is the alpha value [0,255].
 *
 * @see
 * <c>CellGemVideoConvertAttribute</c>, <c>CellGemVideoConvertFlagEnum</c>, 
 * <c>CellGemVideoConvertFormatEnum</c>, <c>cellGemPrepareVideoConvert</c>, <c>cellGemVideoConvertAttributeInitRgba</c>, 
 * <c>cellGemGetVideoConvertSize</c>
*/
static inline void cellGemVideoConvertAttributeInit(CellGemVideoConvertAttribute* retAttr,
	int output_format,
	int conversion_flags,
	float gain,
	float red_gain,
	float green_gain,
	float blue_gain,
	void *buffer_memory,
	void *video_data_out,
	uint8_t alpha)
{
	retAttr->version = CELL_GEM_VERSION;
	retAttr->output_format = output_format;
	retAttr->conversion_flags = conversion_flags;
	retAttr->gain = gain;
	retAttr->red_gain = red_gain;
	retAttr->green_gain = green_gain;
	retAttr->blue_gain = blue_gain;
	retAttr->buffer_memory = buffer_memory;
	retAttr->video_data_out = video_data_out;
	retAttr->alpha = alpha;
}


/** 
 * @brief Initializes the <c>CellGemVideoConvertAttribute</c> structure.
 * 
 * This function initializes the <c>CellGemVideoConvertAttribute</c> structure with defaults for
 * RGBA 640x480 video output.
 * 
 * @param retAttr Attribute structure to initialize
 * @param rgba_out Pointer to 640x480x4 RGBA output buffer
 * @param frame_buffer Optional 640x480x1 128-byte-aligned buffer for adding previous frame to improve image quality
 *
 * @see
 * <c>cellGemPrepareVideoConvert</c>, <c>cellGemVideoConvertAttributeInit</c>, 
 * <c>CellGemVideoConvertAttribute</c>, <c>cellGemGetVideoConvertSize</c>
*/
static inline void cellGemVideoConvertAttributeInitRgba(CellGemVideoConvertAttribute* retAttr,
	void *rgba_out, void *frame_buffer)
{
	int conversion_flags = CELL_GEM_AUTO_WHITE_BALANCE|CELL_GEM_GAMMA_BOOST | CELL_GEM_FILTER_OUTLIER_PIXELS;
	if(frame_buffer) conversion_flags |= CELL_GEM_COMBINE_PREVIOUS_INPUT_FRAME;
	cellGemVideoConvertAttributeInit(retAttr,
		CELL_GEM_RGBA_640x480,
		conversion_flags,
		1.0f,1.0f,1.0f,1.0f,
		frame_buffer,
		rgba_out,
		255);
}


/** 
 * @brief Initializes libgem video output mode.
 * 
 * If this function is never called, video output processing is disabled.
 * Call this function whenever the video output settings change.
 * 
 * @param attr Attribute structure initialized with <c>cellGemVideoConvertAttributeInit()</c>
 *
 * @retval CELL_OK   Successful completion.
 * @retval CELL_GEM_ERROR_UNINITIALIZED   libgem is not initialized.
 * @retval CELL_GEM_ERROR_INVALID_PARAMETER  One of the parameters is invalid.
 * @retval CELL_GEM_ERROR_INVALID_ALIGNMENT  The buffer alignment is invalid.
 * @retval CELL_GEM_ERROR_RESOURCE_ALLOCATION_FAILED   Failed to allocate SPURS workload.
 *
 * @see
 * <c>CellGemVideoConvertAttribute</c>, <c>cellGemVideoConvertAttributeInit</c>, 
 * <c>cellGemVideoConvertAttributeInitRgba</c>
*/
int32_t cellGemPrepareVideoConvert(const CellGemVideoConvertAttribute* attr);

/** 
 * @brief Starts conversion of video obtained from the camera.
 * 
 * This function starts asynchronously converting a video frame from 
 * <c>cellCameraReadEx()</c>.  It uses a separate job chain from the
 * <c>cellGemUpdateStart()</c> and <c>cellGemUpdateFinish()</c>, so 
 * it can be called independently.
 * 
 * @param   video_frame   Values from <c>CellCameraInfoEx.<i>buffer</i></c> (from <c>camera.h</c>)
 *
 * @retval CELL_OK   Successful completion.
 * @retval CELL_GEM_ERROR_UNINITIALIZED  libgem is not initialized.
 * @retval CELL_GEM_ERROR_INVALID_PARAMETER  The <c><i>video_frame</i></c> data is not valid.
 * @retval CELL_GEM_ERROR_INVALID_ALIGNMENT  The buffer alignment is invalid.
 * @retval CELL_GEM_ERROR_CONVERT_NOT_FINISHED  The previous call to <c>cellGemConvertVideoFinish()</c> is not complete.
 *
 * @see
 * <c>cellGemConvertVideoFinish</c>, <c>CellCameraInfoEx</c>, <c>cellCameraReadEx</c> (see the <i>libcamera Reference</i>)
*/
int32_t cellGemConvertVideoStart(const void *video_frame);

/** 
 * @brief Waits for asynchronous SPU processing of <c>cellGemConvertVideoStart()</c> to complete.
 * 
 * This function waits for asynchronous SPU processing of <c>cellGemConvertVideoStart()</c> to complete.
 * The output image buffer is only valid after calling this function.
 * 
 * @retval CELL_OK   Successful completion.
 * @retval CELL_GEM_ERROR_UNINITIALIZED  libgem is not initialized.
 * @retval CELL_GEM_ERROR_CONVERT_NOT_STARTED  <c>cellGemConvertVideoStart()</c> wasn't called yet.

 * @see
 * <c>cellGemConvertVideoStart</c>
*/
int32_t cellGemConvertVideoFinish();

/** 
 * @brief Starts processing video obtained from the camera.
 * 
 * This function starts asynchronously processing a video frame from 
 * <c>cellCameraReadEx()</c> to track position of connected motion controllers.
 * 
 * @param   camera_frame   Values from <c>CellCameraInfoEx.<i>buffer</i></c> (from <c>camera.h</c>)
 * @param   timestamp      Result from <c>cellCameraReadEx()</c>
 *
 * @retval CELL_OK   Successful completion.
 * @retval CELL_GEM_ERROR_UNINITIALIZED  libgem is not initialized.
 * @retval CELL_GEM_ERROR_INVALID_PARAMETER  The <c><i>camera_frame</i></c> data is not valid.
 * @retval CELL_GEM_ERROR_INVALID_ALIGNMENT  The buffer alignment is invalid.
 * @retval CELL_GEM_ERROR_UPDATE_NOT_FINISHED  The previous call to <c>cellGemUpdateFinish()</c> has not completed.
 * @see
 * <c>CellCameraInfoEx</c>, <c>cellCameraReadEx</c> (see the <i>libcamera Reference</i>)
*/
int32_t cellGemUpdateStart(const void *camera_frame, system_time_t timestamp);


/** 
 * @brief Waits for asynchronous SPU processing of <c>cellGemUpdateStart()</c> to complete.
 * 
 * This function waits for asynchronous SPU processing of <c>cellGemUpdateStart()</c> to complete.
 * It must be called for new camera data or inertial sensor data to be used.
 * 
 * @retval CELL_OK   Successful completion.
 * @retval CELL_GEM_ERROR_UNINITIALIZED   libgem is not initialized.
 * @retval CELL_GEM_ERROR_UPDATE_NOT_STARTED  <c>cellGemUpdateStart()</c> was not called yet.
 * @see
 * <c>cellGemUpdateStart</c>
*/
int32_t cellGemUpdateFinish();

/** 
 * @brief Retrieves libgem status flags.
 * 
 * This function retrieves the latched libgem state including calibration feedback
 * and runtime warnings. The flags are described in the <i>libgem Overview</i>.
 * 
 * @param   gem_num     The motion controller from which the data is retrieved (0 - 3)
 * @param   flags		Pointer to variable receiving the flags. See flags starting with <c>CELL_GEM_ALL_FLAGS</c>.
 *
 * @retval CELL_OK   Successful completion.
 * @retval CELL_GEM_ERROR_UNINITIALIZED  libgem is not initialized.
 * @retval CELL_GEM_ERROR_INVALID_PARAMETER  One of the parameters is not valid.
 * @see
 *  <c>cellGemClearStatusFlags</c> 
*/
int32_t cellGemGetStatusFlags(int gem_num, uint64_t* flags);

/** 
 * @brief Clears some or all of libgem status flags.
 * 
 * This function clears libgem status flags. The flags to be cleared must be marked
 * with a binary 1 in the mask. To clear all flags use <c>CELL_GEM_ALL_FLAGS</c>.
 * 
 * @param   gem_num     The motion controller from which the data is retrieved (0 - 3)
 * @param   mask		Mask indicating which flags are to be cleared
 *
 * @retval CELL_OK   Successful completion.
 * @retval CELL_GEM_ERROR_UNINITIALIZED  libgem is not initialized.
 * @retval CELL_GEM_ERROR_INVALID_PARAMETER  One of the parameters is invalid.
 * @see
 *  <c>cellGemGetStatusFlags</c> 
*/
int32_t cellGemClearStatusFlags(int gem_num, uint64_t mask);

/** 
 * @brief Retrieves the motion controller state at a specified point in time.
 * 
 * Retrieves the state of the motion controller combining the results of <c>cellGemUpdateStart()</c>, <c>cellGemUpdateFinish()</c>, and data from
 * inertial sensors. 
 * 
 * @param   gem_num     The motion controller from which the data is retrieved (0 - 3)
 * @param   flag	      The type of position data to retrieve:
 *			   <ul><li><c>CELL_GEM_STATE_FLAG_CURRENT_TIME</c>: State matched to the current system time 
 *                                 when <c>cellGemGetState()</c> is called.  This is the best choice to use 
 *                                 when the video is not displayed.</li>
 *			   <li><c>CELL_GEM_STATE_FLAG_LATEST_IMAGE_TIME</c>: State matched to time the sphere was 
 *                                 imaged in the latest camera frame. This is the best choice to use 
 *																 when the video is displayed.</li>
 *			   <li><c>CELL_GEM_STATE_FLAG_TIMESTAMP</c>: State matched to a particular timestamp.</li>

* @param   time_parameter Information to determine the time for which the state is retrieved. The value of this parameter depends on the value of <c><i>flag</i></c>:
 *			   <ul><li><c>CELL_GEM_STATE_FLAG_CURRENT_TIME</c>: <c><i>time_parameter</i></c> is the time offset (in microseconds) relative to the current time. A negative value means earlier in time. To set the average time offset, use <c>CELL_GEM_LATENCY_OFFSET</c>.</li>
 *			   <li><c>CELL_GEM_STATE_FLAG_LATEST_IMAGE_TIME</c>: <c><i>time_parameter</i></c> is the time offset (in microseconds) relative to the latest image time. A negative value means earlier in time.</li>
 *			   <li><c>CELL_GEM_STATE_FLAG_TIMESTAMP</c>: <c><i>time_parameter</i></c> is the time for which the state is retrieved.

 * @param   gem_state   The latest motion controller state organized as a <c>CellGemState</c> structure, for return.
 *
 * @retval CELL_OK   Successful completion.
 * @retval CELL_GEM_ERROR_UNINITIALIZED  libgem is not initialized.
 * @retval CELL_GEM_ERROR_INVALID_PARAMETER  One of the parameters is invalid.
 * @retval CELL_GEM_NOT_CONNECTED   The specified motion controller has not established a connection to the PlayStation&reg;3.
 * @retval CELL_GEM_SPHERE_NOT_CALIBRATED   The sphere of the specified motion controller is not calibrated.
 * @retval CELL_GEM_SPHERE_CALIBRATING   The sphere of the specified motion controller is in the process of calibrating.
 * @retval CELL_GEM_COMPUTING_AVAILABLE_COLORS   libgem is determining the available tracking hues for the sphere LED.
 * @retval CELL_GEM_HUE_NOT_SET   The hue of the sphere is not set.
 * @retval CELL_GEM_NO_VIDEO   No usable video is specified.
 * @retval CELL_GEM_TIME_OUT_OF_RANGE   No state is available at the specified time.
 * 
 * @see
 * <c>CellGemState</c>, <c>cellGemUpdateFinish</c>, <c>cellGemUpdateStart</c>
*/
int32_t	cellGemGetState(uint32_t gem_num, uint32_t flag, system_time_t time_parameter, CellGemState* gem_state);


/** 
 * @brief Retrieves the low-level motion controller inertial sensor state.
 * 
 * This function retrieves recent inertial sensor readings.
 * Sensor readings are in motion controller coordinates (not world coordinates). Refer to the coordinate description at <c>CellGemState</c>.
 *
 * Using <c>CELL_GEM_INERTIAL_STATE_FLAG_LATEST</c> repeatedly does not guarantee that all the data will be retrieved.
 * To ensure all available inertial data is read, use something like the following code:
 * @code
 * CellGemInertialState inertial_state;
 * static system_time_t inertial_time=0;
 * while (CELL_OK==cellGemGetInertialState(0,CELL_GEM_INERTIAL_STATE_FLAG_NEXT, inertial_time, &inertial_state) {
 *   // use/copy/record inertial_state here
 *   inertial_time = inertial_state.timestamp;
 * }
 * @endcode
 *
 * @note
 * The accelerometer position relative to the center of the sphere is 
 * provided by <c>cellGemGetAccelerometerPositionInDevice()</c>.
 * 
 * @note
 * We generally recommend that you use <c>cellGemGetState()</c> to retrieve the high-level motion controller state 
 * rather than use this low-level inertial data.
 *
 * @param    gem_num	     The motion controller from which the data is retrieved (0 - 3).
 * @param    flag	         The type of inertial data to retrieve:
 *			   <ul><li><c>CELL_GEM_INERTIAL_STATE_FLAG_LATEST</c>: Latest available inertial data
 *                                 when <c>cellGemGetInertialState()</c> is called.</li>
 *			   <li><c>CELL_GEM_INERTIAL_STATE_FLAG_PREVIOUS</c>: Inertial data that precedes specified system timestamp.</li>
 *			   <li><c>CELL_GEM_INERTIAL_STATE_FLAG_NEXT</c>: Inertial data that is subsequent to the specified system timestamp.</li>
 * @param    timestamp		       System timestamp.
 * @param    gem_inertial_state	   The sensor readings organized as a <c>CellGemInertialState</c> structure.
 *
 * @retval CELL_OK   Successful completion.
 * @retval CELL_GEM_TIME_OUT_OF_RANGE No inertial data is available for the specified timestamp.
 * @retval CELL_GEM_ERROR_UNINITIALIZED  libgem is not initialized.
 * @retval CELL_GEM_ERROR_INVALID_PARAMETER  One of the parameters is invalid.
 * @see
 * <c>CellGemInertialState</c>, <c>CellGemState</c>, <c>cellGemGetState</c>, <c>cellGemGetAccelerometerPositionInDevice</c>
*/
 int32_t	cellGemGetInertialState(uint32_t gem_num, uint32_t flag, system_time_t timestamp, CellGemInertialState* gem_inertial_state);


/** 
 * @brief Retrieves accelerometer position offset from the sphere.
 * 
 * This function retrieves the accelerometer position as it is offset from the center of the sphere, in motion controller coordinates.
 * Refer to the coordinate description at <c>CellGemState</c>.
 * 
 * @param    gem_num	    The motion controller for which the data is retrieved (0 - 3)
 * @param    pos		    Position offset (in mm)
 *
 * @retval CELL_OK   Successful completion.
 * @retval CELL_GEM_ERROR_UNINITIALIZED  libgem is not initialized.
 * @retval CELL_GEM_ERROR_INVALID_PARAMETER  One of the parameters is invalid.
 * @see
 * <c>CellGemState</c>, <c>CellGemInertialState</c>, <c>cellGemGetInertialState</c>
*/
int32_t cellGemGetAccelerometerPositionInDevice(uint32_t gem_num, vec_float4* pos);


/** 
 * @brief Retrieves the state from vision processing of the latest image.
 * 
 * Retrieves the tracking result from vision processing of the latest image performed with 
 * <c>cellGemUpdateStart()</c> and <c>cellGemUpdateFinish()</c>.
 * 
 * @param    gem_num	  The motion controller for which the data is retrieved (0 - 3)
 * @param    gem_image_state	  The latest image state organized as a <c>CellGemImageState</c> structure
 *
  * @retval CELL_OK   Successful completion.
 * @retval CELL_GEM_ERROR_UNINITIALIZED  libgem is not initialized.
 * @retval CELL_GEM_ERROR_INVALID_PARAMETER  One of the parameters is invalid.
 * @see
 * <c>CellGemImageState</c>, <c>cellGemUpdateFinish</c>, <c>cellGemUpdateStart</c>
*/
int32_t cellGemGetImageState(uint32_t gem_num, CellGemImageState* gem_image_state);

/** 
 * @brief Retrieves the state of the camera.
 * 
 * Retrieves the state specifying current camera parameters.
 * 
 * @param    gem_camera_state	  The latest camera state organized as a <c>CellGemImageState</c> structure.
 *
 * @retval CELL_OK   Successful completion.
 * @retval CELL_GEM_ERROR_UNINITIALIZED  libgem is not initialized.
 * @retval CELL_GEM_ERROR_INVALID_PARAMETER  One of the parameters is invalid.
 * @see
 * <c>CellGemCameraState</c>
*/
int32_t cellGemGetCameraState(CellGemCameraState* gem_camera_state);


/** 
 * @brief Forces the sphere to an exact RGB color.
 * 
 * This function allows you to force the sphere to be a specific color.
 * Positional tracking is disabled if the color is set this way. Also, the forced
 * color may worsen tracking for other motion controllers if multiple motion controllers are in use.
 * 
 * @param    gem_num	    The motion controller for which the color is set (0 - 3)
 * @param    r				Red component of color, 0.0 to 1.0
 * @param    g				Green component of color, 0.0 to 1.0
 * @param    b				Blue component of color, 0.0 to 1.0
 *
 * @retval CELL_OK   Successful completion.
 * @retval CELL_GEM_ERROR_UNINITIALIZED  libgem is not initialized.
 * @retval CELL_GEM_ERROR_INVALID_PARAMETER  One of the parameters is invalid.
 * @see
 * <c>cellGemGetRGB</c>, <c>cellGemTrackHues</c>
*/
int32_t cellGemForceRGB(uint32_t gem_num, float r, float g, float b);


/** 
 * @brief Produces a vibration in the motion controller.
 * 
 * This function allows you to produce vibration in the motion controller to provide
 * feedback to the user.
 * 
 * @param    gem_num	    The motion controller for which the vibration is set (0 - 3)
 * @param    rumble         Intensity of the vibration, from 0 to 255
 *
 * @retval CELL_OK   Successful completion.
 * @retval CELL_GEM_ERROR_UNINITIALIZED  libgem is not initialized.
 * @retval CELL_GEM_ERROR_INVALID_PARAMETER  One of the parameters is invalid.
 * @see
 * <c>cellGemGetRumble</c>
*/
int32_t cellGemSetRumble(uint32_t gem_num, unsigned char rumble);


/** 
 * @brief Reads information from the peripheral connected to the motion controller's extension connector.
 * 
 * This function allows you to read description information from a device connected
 * to the extension connector on the motion controller.  Note that this function takes ~300 milliseconds
 * to execute when a peripheral is connected to the motion controller. 
 * 
 * @param    gem_num	    The motion controller for which the extension connector information is read (0 - 3)
 * @param    ext_id         ID (type) of the attached peripheral device
 * @param    ext_info       Array of bytes containing device-specific information
 *
 * @retval CELL_OK   Successful completion.
 * @retval CELL_GEM_ERROR_UNINITIALIZED  libgem is not initialized.
 * @retval CELL_GEM_ERROR_INVALID_PARAMETER  One of the parameters is invalid.
 * @retval CELL_GEM_NOT_CONNECTED   The specified motion controller has not established a connection to the PlayStation&reg;3.
 * @retval CELL_GEM_NO_EXTERNAL_PORT_DEVICE   The specified motion controller does not have a peripheral device connected to the extension connector.
*/
int32_t cellGemReadExternalPortDeviceInfo(uint32_t gem_num, unsigned int *ext_id, unsigned char ext_info[CELL_GEM_EXTERNAL_PORT_DEVICE_INFO_SIZE]);

/** 
 * @brief Writes data to the motion controller's extension connector.
 * 
 * This function allows you to write custom data to a peripheral device connected
 * to the extension connector on the motion controller.
 * The data will be sent exactly once. There is no guarantee or handshake
 * to ensure that the data arrives.
 * 
 * @param    gem_num	    The motion controller with the extension connector to which data is written  (0 - 3)

 * @param    data         Array of bytes to be written
 *
 * @retval CELL_OK   Successful completion.
 * @retval CELL_GEM_ERROR_WRITE_NOT_FINISHED	The call was attempted before the previous <c>cellGemWriteExternalPort()</c> call completed.
 * @retval CELL_GEM_ERROR_UNINITIALIZED  libgem is not initialized.
 * @retval CELL_GEM_ERROR_INVALID_PARAMETER  One of the parameters is invalid.
 * @retval CELL_GEM_NOT_CONNECTED   The specified motion controller has not established a connection to the PlayStation&reg;3.
*/
int32_t cellGemWriteExternalPort(uint32_t gem_num, unsigned char data[CELL_GEM_EXTERNAL_PORT_OUTPUT_SIZE]);


/** 
 * @brief Retrieves the value of the motion controller vibration.
 * 
 * This function allows you to retrieve the vibration setting.
 * 
 * @param    gem_num	    The motion controller for which the vibration setting is retrieved (0 - 3)
 * @param    rumble         Intensity of the vibration, from 0 to 255
 *
 * @retval CELL_OK   Successful completion.
 * @retval CELL_GEM_ERROR_UNINITIALIZED  libgem is not initialized.
 * @retval CELL_GEM_ERROR_INVALID_PARAMETER  One of the parameters is invalid.
 * @see
 * <c>cellGemSetRumble</c>
*/
int32_t cellGemGetRumble(uint32_t gem_num,unsigned char* rumble);


/** 
 * @brief Retrieves the exact RGB color of the sphere LED.
 * 
 * This function retrieves the current color of the motion controller sphere.
 * 
 * @param    gem_num	The motion controller for which the color is retrieved (0 - 3)
 * @param    r		Pointer to variable receiving red component of color
 * @param    g		Pointer to variable receiving green component of color
 * @param    b		Pointer to variable receiving blue component of color
 *
 * @retval CELL_OK   Successful completion.
 * @retval CELL_GEM_ERROR_UNINITIALIZED  libgem is not initialized.
 * @retval CELL_GEM_ERROR_INVALID_PARAMETER  One of the parameters is invalid.
 * @see
 * <c>cellGemForceRGB</c>, <c>cellGemTrackHues</c>
*/
int32_t cellGemGetRGB(uint32_t gem_num, float* r, float* g, float* b);


/** 
 * @brief Retrieves the current tracked hue.
 * 
 * This function retrieves the tracking hue of the motion controller.
 * 
 * @param    gem_num	The motion controller for which the hue is retrieved (0 - 3)
 * @param    hue		Pointer to a variable receiving the hue
 *
 * @retval CELL_OK   Successful completion.
 * @retval CELL_GEM_ERROR_NOT_A_HUE    The motion controller is not currently tracked and the received hue is invalid.
 * @retval CELL_GEM_ERROR_UNINITIALIZED   libgem is not initialized.
 * @retval CELL_GEM_ERROR_INVALID_PARAMETER  One of the parameters is invalid.
 * @see
 * <c>cellGemTrackHues</c>
*/
int32_t cellGemGetTrackerHue(uint32_t gem_num, unsigned int* hue);


/** 
 * @brief Converts a point in HSV space to RGB space using the libgem converter.
 * 
 * This function is provided because there is no single standard method for HSV to RGB conversion.
 * This version of the library treats HSV as a hexacone. (See http://en.wikipedia.org/wiki/Hue as of August 1, 2010. Note that pages may have been moved or modified.)
 * Using this function allows the application to display hues using RGB values.
 * This function can also be used to convert hues for a <c>cellGemForceRGB()</c> call.
 * 
 * @param    h		Input hue angle. Values outside of [0-359] will be wrapped.
 * @param    s		Input saturation, range [0-1]
 * @param    v		Input value (brightness), range [0-1] 
 * @param    r		Pointer to variable receiving output red component of color
 * @param    g		Pointer to variable receiving output green component of color
 * @param    b		Pointer to variable receiving output blue component of color
 *
 * @retval CELL_OK   Successful completion.
 * @retval CELL_GEM_ERROR_INVALID_PARAMETER  One of the parameters is invalid.
*/
int32_t cellGemHSVtoRGB(float h, float s, float v, float* r, float* g, float* b);


/** 
 * @brief Queries the library for whether or not a hue will be trackable.
 * 
 * This function checks if a given hue will succeed in tracking 
 * without actually changing the sphere color to that hue.
 * Note that even though a hue may be trackable, it may be too similar to a hue
 * used by another controller. This function is useful for noting which colors
 * in the user environment are unavailable.
 * The results from this function may change after a <c>cellGemCalibrate()</c> or
 * <c>cellGemPrepareCamera()</c> call.
 * 
 * @param    hue		The hue to check (0-359 degrees inclusive)
 *
 * @return
 * <c>1</c> if the hue is trackable. 
 * <c>0</c> if the hue is not trackable.
 * @note
 * The function always returns 0 during sphere calibration.
 * @see
 * <c>cellGemTrackHues</c>
*/
int32_t cellGemIsTrackableHue(unsigned int hue);

/** 
 * @brief Queries the library for all hues that are trackable.
 * 
 * This function returns whether hues are trackable given the current user environment
 * without changing the sphere color. It performs the same purpose 
 * as <c>cellGemIsTrackableHue()</c>, but returns trackable status for all hues, 0 through 359. 
 * It is much faster to use this function than calling <c>cellGemIsTrackableHue()</c> for 
 * each individual hue. 
 *
 * @param    hues		Pointer to an array that is able to hold 360 Boolean results.
 *
 * @retval CELL_OK   Successful completion.
 * @see
 * <c>cellGemIsTrackableHue</c>
*/
int32_t cellGemGetAllTrackableHues(unsigned char* hues);

/** 
 * @brief Identify pixels in an image that match a given hue.
 * 
 * This function uses the internal tracking thresholds to set values in an output 
 * array that correspond to pixels in the PlayStation&reg;Eye camera frame that match a given 
 * hue. This function is useful for noting which colors in the user environment are 
 * unavailable, and for noting what scene items may be producing a conflict. The input camera frame should 
 * be the original Bayer data read from the PlayStation&reg;Eye (the same data that is passed to 
 * <c>cellGemUpdateStart()</c>). The results from this function may change after a <c>cellGemCalibrate()</c>
 * or <c>cellGemPrepareCamera()</c> call.
 *
 * To understand what in the user environment is making a hue untrackable, pass the untrackable
 * hue to this function. The pixels in the function result identify the places in the user environment that 
 * make the hue untrackable.
 *
 * If the sphere is set to the same color as the hue passed to this function, the sphere itself
 * will appear in the function results.
 * 
 * @param   camera_frame   Values from <c>CellCameraInfoEx.<i>buffer</i></c> (from <c>camera.h</c>).
 * @param   hue			The hue for which results are desired (0-359 degrees inclusive).
 * @param   pixels		Array of 640x480 unsigned chars. This array must be 128-byte aligned. Each pixel is set to "0" if it does not match the given hue, or "nonzero" if it does.
 *
 * @retval CELL_OK   Successful completion.
 * @retval CELL_GEM_ERROR_UNINITIALIZED   libgem is not initialized.
 * @retval CELL_GEM_ERROR_INVALID_PARAMETER  One of the parameters is invalid.
 *
 * @note
 * The function always returns 0 during sphere calibration.
 * @see
 * <c>cellGemTrackHues</c>
*/int32_t cellGemGetHuePixels(const void *camera_frame, unsigned int hue, unsigned char *pixels);

/** 
 * @brief Sets controller hues for tracking and begins tracking.
 * 
 * This function attempts to set the controller spheres to the requested
 * colors. If the colors are too close to one another, the function will automatically
 * move them apart until there is no conflict. Each hue can be one of 3 types: 
 * a number (0 to 359) specifying the requested hue, a <c>CELL_GEM_DONT_CARE_HUE</c>, or a <c>CELL_GEM_DONT_TRACK_HUE</c>.
 *
 * A <c>CELL_GEM_DONT_CARE_HUE</c> value sets the controller to the best color for tracking provided
 * that it is not near other, specified controller colors. Controllers set to this value
 * have lower priority during color picking than those with a specified hue.
 *
 * A <c>CELL_GEM_DONT_TRACK_HUE</c> value stops tracking for that controller if it is tracking and 
 * turns the LED off. If the controller is already not tracking, <c>CELL_GEM_DONT_TRACK_HUE</c> does not change
 * the tracking or LED status.
 *
 * The function always succeeds in setting at least some hue so that tracking may continue.
 * In case of a difficult tracking situation with many colors in the user environment, the flag
 * <c>CELL_GEM_FLAG_VERY_COLORFUL_ENVIRONMENT</c> will be raised indicating that tracking may be
 * worse due to too many conflicting colors.
 *
 * The hues that are picked are returned in the <c><i>res_hues</i></c> array.
 * 
 * @param    req_hues	Pointer to the array of four requested hues. [0-359] or flag.
 * @param    res_hues	Pointer to the array of four resulting hues. Not filled if NULL.
 *
 * @retval CELL_OK   Successful completion.
 * @retval CELL_GEM_ERROR_INVALID_PARAMETER  One of the parameters is invalid.
 * @note
 * Passing hue values outside of the 0 to 359 hue range can result in unexpected behavior.
 * @see
 * <c>cellGemIsTrackableHue</c>
*/
int32_t cellGemTrackHues(const unsigned int* req_hues, unsigned int* res_hues);


/** 
 * @brief Retrieves the estimated color of the lighting in the current environment at the current exposure.
 * 
 * This function retrieves the color that the unlit sphere appears to the camera.
 * The actual values are computed during sphere calibration for the last motion controller that was calibrated.
 * These values can be used for game lighting in augmented reality or image white balance.
 * 
 * @param    r		Pointer to a variable receiving the red component of color
 * @param    g		Pointer to a variable receiving the green component of color
 * @param    b		Pointer to a variable receiving the blue component of color
 *
 * @retval CELL_OK   Successful completion.
 * @retval CELL_GEM_ERROR_LIGHTING_NOT_CALIBRATED  Calibration has not occurred for lighting. The values returned are 128,128,128.
 * @retval CELL_GEM_ERROR_UNINITIALIZED   libgem is not initialized.
 * @retval CELL_GEM_ERROR_INVALID_PARAMETER  One of the parameters is invalid.
 * @note
 * <c>CELL_GEM_ERROR_LIGHTING_NOT_CALIBRATED</c> is returned after initial connection and can be corrected by performing 
 * a calibration with <c>cellGemCalibrate()</c> on at least one motion controller.
 * @see
 * <c>cellGemCalibrate</c>
*/
int32_t cellGemGetEnvironmentLightingColor(float* r, float* g, float* b);


/** 
 * @brief Retrieves motion controller connection information.
 * 
 * This function retrieves connection information for all currently connected motion controllers.
 * 
 * @param   info	Motion controller connection information organized as a <c>CellGemInfo</c> structure.
 *
 * @retval CELL_OK   Successful completion.
 * @retval CELL_GEM_ERROR_UNINITIALIZED   libgem is not initialized.
 * @retval CELL_GEM_ERROR_INVALID_PARAMETER  One of the parameters is invalid.
 * @see
 * <c>CellGemInfo</c>
*/
int32_t cellGemGetInfo(CellGemInfo *info);


/** 
 * @brief Resets a motion controller to the state it was in when first connected.
 * 
 * Call this function to reset a motion controller to the state it was in when first connected.
 * This function is automatically called internally upon initial connection.
 * 
 * @param    gem_num	    The motion controller for which the state is set (0 - 3)
 *
 * @retval CELL_OK   Successful completion.
 * @retval CELL_GEM_ERROR_UNINITIALIZED   libgem is not initialized.
 * @retval CELL_GEM_ERROR_INVALID_PARAMETER  One of the parameters is invalid.
 * @see
 * <c>cellGemGetInfo</c>
*/
int32_t cellGemReset(uint32_t gem_num);


/** 
 * @brief Initiates calibration.
 * 
 * This function begins the calibration procedure. The motion controller should be held still during calibration; 
 * however, the system can accommodate minor movements (such as the motion controller held by an unstable hand).
 * The motion controller must be pointed at the PlayStation&reg;Eye when calibrating, as calibration adjusts the
 * camera pitch angle estimate.  If the motion controller is not pointed directly at the PlayStation&reg;Eye, 
 * tracking will not work properly.

 * Calibration also clears the magnetic map entries created by <c>cellGemSetYaw()</c>.
 * 
 * @param    gem_num	    The motion controller to which to apply the setting (0 - 3)
 *
 * @retval CELL_OK   Successful completion.
 * @retval EBUSY     Calibration or camera preparation is already in progress.
 * @retval CELL_GEM_ERROR_UNINITIALIZED   libgem is not initialized.
 * @retval CELL_GEM_ERROR_INVALID_PARAMETER  One of the parameters is invalid.
*/
int32_t cellGemCalibrate(uint32_t gem_num);


/** 
 * @brief Invalidates a motion controller's calibration.
 * 
 * This function invalidates the calibration. It is used to put the motion controller into an uncalibrated
 * state to necessitate recalibration. Use it to signal that a motion controller needs to be recalibrated, perhaps if the system detected a
 * change in the lighting environment during calibration of another motion controller.
 * 
 * @param    gem_num	    The motion controller to which to apply the setting (0 - 3)
 *
 * @retval CELL_OK   Successful completion.
 * @retval CELL_GEM_ERROR_UNINITIALIZED   libgem is not initialized.
 * @retval CELL_GEM_ERROR_INVALID_PARAMETER  One of the parameters is invalid.
*/
int32_t cellGemInvalidateCalibration(int32_t gem_num);


/** 
 * @brief Controls the performance of state filtering
 * 
 * This function sets whether to perform filtering to increase smoothness of 
 * <c>CellGemState</c> output. Enabling filtering can cause some increase of tracking latency. The 
 * default behavior does not perform filtering.
 * 
 * @param    gem_num	    The motion controller for which to apply the setting (0 - 3)
 * @param    flag           Set to 1 to enable state filtering
 *
 * @retval CELL_OK   Successful completion.
 * @retval CELL_GEM_ERROR_UNINITIALIZED   libgem is not initialized.
 * @retval CELL_GEM_ERROR_INVALID_PARAMETER  One of the parameters is invalid.
 * @see 
 * <c>CellGemState</c>, <c>cellGemGetState</c>
*/
int32_t cellGemFilterState(uint32_t gem_num, uint32_t flag);


/** 
 * @brief Controls the magnetometer.
 * 
 * This function controls whether libgem uses data from the magnetometer to correct 
 * <c>CellGemState</c>. The magnetometer is most useful for applications requiring precise
 * yaw angle, such as pointing or shooting. The default behavior does not use the
 * magnetometer; the magnetometer is disabled when <c>cellGemReset()</c> is called with the motion controller connected.
 * 
 * @param    gem_num	The motion controller for which to apply the setting (0 - 3)
 * @param    flag			Set to 1 to enable use of the magnetometer
 *
 * @retval CELL_OK   Successful completion.
 * @retval CELL_GEM_ERROR_UNINITIALIZED  libgem is not initialized.
 * @retval CELL_GEM_NOT_CONNECTED   The specified motion controller has not established a connection to the PlayStation&reg;3.
 * @retval CELL_GEM_NOT_CALIBRATED   The specified motion controller's magnetometer has not been calibrated in XMB, so magnetometer will not be used.
*/
int32_t cellGemEnableMagnetometer(uint32_t gem_num, int flag);


/** 
 * @brief Controls the use of automatic camera pitch angle computation.
 * 
 * This function controls whether the <c>CellGemState</c> computation of the camera
 * pitch angle is the estimation automatically computed by libgem. libgem always computes an
 * estimate of the camera pitch angle; this controls whether or not it is constantly applied.
 * 
 * @param    flag	    Set to 0 to disable automatic update of estimated pitch angle
 *
 * @retval CELL_OK   Successful completion.
 * @retval CELL_GEM_ERROR_UNINITIALIZED  libgem is not initialized.
*/
int32_t cellGemEnableCameraPitchAngleCorrection(int flag);

/** 
 * @brief Specifies the current direction of the <c>z</c> axis of the motion controller
 * in world space.
 * 
 * This function allows setting the motion controller&#8216;s <c>z</c> direction to correct the 
 * yaw angle.  It can be used to manually correct yaw drift while the motion controller is pointing at the camera 
 * or other known target.  
 *
 * The primary use for this function is to call it when the motion controller is visible 
 * and known to be pointed directly at the camera.  In such a case, the <c>z</c> direction is
 * the motion controller position as returned in <c>CellGemState</c> (see any of the samples).
 * 
 * When the magnetometer correction is enabled, <c>cellGemSetYaw()</c> adds a reference point to a
 * temporary magnetic map of the environment.  This reference point allows for more precise angle tracking
 * near this location.  If the magnetometer is enabled and <c>cellGemSetYaw()</c> is called, the controller
 * must be kept nearly stationary for 0.5 seconds so that a magnetic reading can be taken.  If
 * successful, the sphere LED will flicker off for 30 milliseconds to signify a magnetic reference was added.
 * Refer to the coordinate description in <c>CellGemState</c>.
 * 
 * @param    gem_num	    The motion controller for which to apply the setting (0 - 3)
 * @param    z_direction    The <c>z</c> direction of the motion controller in world coordinates
 *
 * @retval CELL_OK   Successful completion.
 * @retval CELL_GEM_ERROR_UNINITIALIZED   libgem is not initialized.
 * @retval CELL_GEM_ERROR_INVALID_PARAMETER  One of the parameters is invalid.
 * @see
 * <c>CellGemState</c>
*/
int32_t cellGemSetYaw(uint32_t gem_num, vec_float4 z_direction);


#ifdef __cplusplus
}
#endif /* __cplusplus */
#endif
