/* SCE CONFIDENTIAL
PlayStation(R)3 Programmer Tool Runtime Library 475.001
 * Copyright (C) 2010 Sony Computer Entertainment Inc.
 * All Rights Reserved.
 */
#ifndef __CELL_SPURS_TYPES_H__
#define __CELL_SPURS_TYPES_H__ 1

/* The C Standard Library */
#include <stdint.h>
#include <stdbool.h>

/* SPU C/C++ Language Extention */
#ifdef __SPU__
#include <spu_intrinsics.h>
#endif /* __SPU__ */

/* For types provided by Cell OS Lv-2 */
#include <sys/sys_types.h>

#define CELL_SPURS_MAX_SPU			8
#define CELL_SPURS_NAME_MAX_LENGTH	15

/*
 * workload
 */
#define CELL_SPURS_MAX_WORKLOAD		16
#define CELL_SPURS_MAX_WORKLOAD1	16
#define CELL_SPURS_MAX_WORKLOAD2	32
#define CELL_SPURS_MAX_PRIORITY		16
typedef unsigned CellSpursWorkloadId;
typedef struct {
	uint64_t _unused0;
	uint32_t _unused1;
	uint32_t flag;						/* zero or not */
} __attribute__((aligned(16))) CellSpursWorkloadFlag;

/*
 * SPURS Local Storage layout
 */
#define CELL_SPURS_INTERRUPT_VECTOR		0x000
#define CELL_SPURS_LOCK_LINE			0x080
#define CELL_SPURS_KERNEL_CONTEXT		0x100

/*
 * DMA tag used by the SPURS kernel and modules
 */
#define CELL_SPURS_KERNEL_DMA_TAG_ID	31

/* for backward compatibility */
#define CELL_SPURS_INTERUPT_VECTOR		0x000

/* SPU thread event port */
#define	CELL_SPURS_PORT_RANGE_END			63

#define	CELL_SPURS_STATIC_PORT_NUM			16
#define	CELL_SPURS_STATIC_PORT_RANGE_BOTTOM	(CELL_SPURS_STATIC_PORT_NUM - 1)

#define	CELL_SPURS_DYNAMIC_PORT_RANGE_TOP	(CELL_SPURS_STATIC_PORT_NUM)
#define	CELL_SPURS_DYNAMIC_PORT_RANGE_BOTTOM	CELL_SPURS_PORT_RANGE_END
#define	CELL_SPURS_DYNAMIC_PORT_NUM			(CELL_SPURS_DYNAMIC_PORT_RANGE_BOTTOM - CELL_SPURS_STATIC_PORT_BOTTOM)

/* for cellSpursGetInfo() */
typedef struct CellSpursInfo {
	int nSpus;
	int spuThreadGroupPriority;
	int ppuThreadPriority;
	bool exitIfNoWork;
	bool spurs2;
	uint8_t	__padding24[2];
	void     *traceBuffer;
	uint32_t __padding32;
	uint64_t traceBufferSize;
	uint32_t traceMode;

	sys_spu_thread_group_t  spuThreadGroup;
	sys_spu_thread_t        spuThreads[CELL_SPURS_MAX_SPU];
	__extension__ union {
		sys_ppu_thread_t        spursHandelerThread0;
		sys_ppu_thread_t        spursHandlerThread0;
	};
	__extension__ union {
		sys_ppu_thread_t        spursHandelerThread1;
		sys_ppu_thread_t        spursHandlerThread1;
	};

	char namePrefix[CELL_SPURS_NAME_MAX_LENGTH+1];
	size_t namePrefixLength;

	uint32_t deadlineMissCounter;
	uint32_t deadlineMeetCounter;

	uint8_t padding[280
					- sizeof(int) * 3
					- sizeof(bool) * 2
					- sizeof(uint8_t) * 2
					- sizeof(void*)
					- sizeof(uint32_t)
					- sizeof(uint64_t)
					- sizeof(uint32_t)
					- sizeof(sys_spu_thread_group_t)
					- sizeof(sys_spu_thread_t) * CELL_SPURS_MAX_SPU
					- sizeof(sys_ppu_thread_t) * 2
					- sizeof(char) * (CELL_SPURS_NAME_MAX_LENGTH+1)
					- sizeof(size_t)
					- sizeof(uint32_t) * 2
			];
} CellSpursInfo;

/*
 * SPURS instance
 */

#define CELL_SPURS_ATTRIBUTE_ALIGN	8
#define CELL_SPURS_ATTRIBUTE_SIZE	512

typedef struct CellSpursAttribute {
	unsigned char	skip[CELL_SPURS_ATTRIBUTE_SIZE];
} __attribute__((aligned(CELL_SPURS_ATTRIBUTE_ALIGN))) CellSpursAttribute;

#ifdef __cplusplus
namespace cell {
	namespace Spurs {
		class SpursAttribute : public CellSpursAttribute {
		public:
			static int initialize(SpursAttribute *attribute, int nSpus, int spuPriority, int ppuPriority,  bool exitIfNoWork);
			int setMemoryContainerForSpuThread(sys_memory_container_t container);
			int setNamePrefix(const char* name,size_t size);
			int setSpuThreadGroupType(int type);
			int enableSpuPrintfIfAvailable();
			int enableSystemWorkload(const uint8_t priority[8], unsigned int maxSpu, const bool isPreemptible[8]);
		};
	}
}
#endif // __cplusplus

#define CELL_SPURS_ALIGN			128
#define CELL_SPURS_SIZE				4096
#define CELL_SPURS_SIZE2			8192

typedef struct CellSpurs {
	unsigned char skip[CELL_SPURS_SIZE];
} __attribute__((aligned(CELL_SPURS_ALIGN))) CellSpurs;

#ifdef __cplusplus
#include <cell/spurs/exception_common_types.h> // CellSpursExceptionEventHandler

namespace cell {
	namespace Spurs {
		class Spurs : public CellSpurs {
		public:
			static const uint32_t kMaxWorkload     = CELL_SPURS_MAX_WORKLOAD;
			static const uint32_t kMaxPriority     = CELL_SPURS_MAX_PRIORITY;
			static const uint32_t kNameMaxLength   = CELL_SPURS_NAME_MAX_LENGTH;
			static const uint32_t kAlign           = CELL_SPURS_ALIGN;
			static const uint32_t kSize            = CELL_SPURS_SIZE;
			static const uint32_t kInterruptVector = CELL_SPURS_INTERRUPT_VECTOR;
			static const uint32_t kLockLine        = CELL_SPURS_LOCK_LINE;
			static const uint32_t kKernelDmaTagId  = CELL_SPURS_KERNEL_DMA_TAG_ID;

			static int initialize(Spurs* spurs, unsigned nSpus, int spuPriority, int ppuPriority, bool exitIfNoWork);
			static int initialize(Spurs* spurs, const CellSpursAttribute* attr);
			static int initializeWithAttribute(Spurs* spurs, const CellSpursAttribute* attr);
			int finalize();
			int wakeUp();
			int getNumSpuThread(unsigned *nThreads);
			int setMaxContention(CellSpursWorkloadId wid,  unsigned int maxContention);
			int setPriorities(CellSpursWorkloadId wid, const uint8_t priorities[CELL_SPURS_MAX_SPU]);
			int setPriority(CellSpursWorkloadId wid, unsigned int spu, unsigned int priority);
			int getSpuThreadId(sys_spu_thread_t* thr, unsigned *nThreads);
			int getSpuThreadGroupId(sys_spu_thread_group_t* thr_group);
			int getInfo(CellSpursInfo *info);
			int attachLv2EventQueue(sys_event_queue_t queue, uint8_t* port, int isDynamic);
			int detachLv2EventQueue(uint8_t port);
			int enableExceptionEventHandler(bool flag);
			int setExceptionEventHandler(CellSpursWorkloadId id, CellSpursExceptionEventHandler eaHandler, void* arg);
			int unsetExceptionEventHandler(CellSpursWorkloadId id);
			int setGlobalExceptionEventHandler(CellSpursGlobalExceptionEventHandler eaHandler, void* arg);
			int unsetGlobalExceptionEventHandler();
			int setPreemptionVictimHints(const bool isPreemptible[8]);
		};
	}
}

struct CellSpurs2 : public cell::Spurs::Spurs {
	unsigned char skip[CELL_SPURS_SIZE2 - CELL_SPURS_SIZE];
} __attribute__((aligned(CELL_SPURS_ALIGN)));

namespace cell {
	namespace Spurs {
		class Spurs2 : public CellSpurs2 {
		public:
			static const uint32_t kMaxWorkload     = CELL_SPURS_MAX_WORKLOAD2;
			static const uint32_t kSize            = CELL_SPURS_SIZE2;

			static int initialize(Spurs2* spurs2, const CellSpursAttribute* attr);
			static int initializeWithAttribute2(Spurs2* spurs2, const CellSpursAttribute* attr);
		};
	}
}

#else  // __cplusplus

typedef struct CellSpurs2 {
	unsigned char skip[CELL_SPURS_SIZE2];
} __attribute__((aligned(CELL_SPURS_ALIGN))) CellSpurs2;

#endif // __cplusplus

#endif /* __CELL_SPURS_TYPES_H__ */
/*
 * Local Variables:
 * mode: C
 * c-file-style: "stroustrup"
 * tab-width: 4
 * End:
 * vim:sw=4:sts=4:ts=4
 */
